//
//  TEduBoardResourceController.h
//  TEduBoard
//
//  Created by 张小桥 on 2021/12/29.
//  Copyright © 2021 kennethmiao. All rights reserved.
//

#import <Foundation/Foundation.h>


NS_ASSUME_NONNULL_BEGIN

/**
 * @ingroup def
 * @brief 资源加载配置项
*/
@interface TEduBoardResourceConfig : NSObject
@property (nonatomic, assign) NSInteger sdkAppId;        ///< 应用标识
@property (nonatomic, strong) NSString* userId;         ///< 用户标识
@end

/**
 * @ingroup def
 * @brief 元素类型
*/
typedef NS_ENUM(NSInteger, TEduBoardPreloadResourceType)
{
    TEDU_BOARD_PRELOAD_RESOURCE_TYPE_IMAGE                    =   1,      ///< 图片
    TEDU_BOARD_PRELOAD_RESOURCE_TYPE_MEDIA                    =   2,      ///< 多媒体（mp3, mp4）
};

/**
 * @brief 资源加载回调
 * @param url                加载的资源url
 * @param progress           下载进度，正常情况下为 [0, 100], 出错时，会返回progress -1
 * @param error              下载发生错误时的错误代码
 */
typedef void (^TEduResourceLoadCallback)(NSString *url, NSInteger progress, NSError *error);


@interface TEduBoardResourceController : NSObject
/**
 * @brief 获取TEduBoardResourceController实例
 */
+ (instancetype)sharedInstance;

/**
 * @brief 设置资源加载config
 */
+ (void)setConfig:(TEduBoardResourceConfig *)config;


/**
 * @brief 用于课件预加载，减少课件的加载时间，默认缓存259200秒（3天）；超出过期时间后，本地预加载资源会失效。
 * @param zipUrl                课件zip地址
 * @warning loadResource之前请先调用 `[TEduBoardResourceController setConfig:]`
 * @warning iOS 11.4、macOS 11.0以上系统有效
 */
+ (BOOL)loadResource:(NSString *)zipUrl;

/**
 * @brief 用于课件预加载，减少课件的加载时间
 * @param zipUrl                课件zip地址
 * @param expireTime            过期时间，单位秒，-1 表示永久有效，默认259200秒（3天）；超出过期时间后，本地预加载资源会失效。
 * @warning loadResource之前请先调用 `[TEduBoardResourceController setConfig:]`
 * @warning iOS 11.4、macOS 11.0以上系统有效
 */
+ (BOOL)loadResource:(NSString *)zipUrl expireTime:(NSInteger)expireTime;


/**
 * @brief 用于课件预加载，减少课件的加载时间
 * @param zipUrl                课件zip地址
 * @param expireTime            过期时间，单位秒，-1 表示永久有效，默认259200秒（3天）；超出过期时间后，本地预加载资源会失效。
 * @param callBack              资源加载回调
 * @warning loadResource之前请先调用 `[TEduBoardResourceController setConfig:]`
 * @warning iOS 11.4、macOS 11.0以上系统有效
 */
+ (BOOL)loadResource:(NSString *)zipUrl expireTime:(NSInteger)expireTime callback:(nullable TEduResourceLoadCallback)callBack;

/**
 * @brief 加载自定义资源，默认缓存259200秒（3天）；超出过期时间后，本地预加载资源会失效。
 * @param url                           资源url,请使用Https协议的网络资源url
 * @param resouceType                   具体参考`TEduBoardPreloadResourceType`
 * @warning loadCustomResource之前请先调用 `[TEduBoardResourceController setConfig:]`
 * @warning iOS 11.4、macOS 11.0以上系统有效
 */
+ (BOOL)loadCustomResource:(NSString *)url resouceType:(TEduBoardPreloadResourceType)resouceType;

/**
 * @brief 加载自定义资源
 * @param url                           资源url,请使用Https协议的网络资源url
 * @param resouceType                   具体参考`TEduBoardPreloadResourceType`
 * @param expireTime                    过期时间，单位秒，-1 表示永久有效
 * @warning loadCustomResource之前请先调用 `[TEduBoardResourceController setConfig:]`
 * @warning iOS 11.4、macOS 11.0以上系统有效
 */
+ (BOOL)loadCustomResource:(NSString *)url resouceType:(TEduBoardPreloadResourceType)resouceType expireTime:(NSInteger)expireTime;

/**
 * @brief 加载自定义资源
 * @param url                           资源url,请使用Https协议的网络资源url
 * @param resouceType                   具体参考`TEduBoardPreloadResourceType`
 * @param expireTime                    过期时间，单位秒，-1 表示永久有效
 * @param callBack                      资源加载回调
 * @warning loadCustomResource之前请先调用 `[TEduBoardResourceController setConfig:]`
 * @warning iOS 11.4、macOS 11.0以上系统有效
 */
+ (BOOL)loadCustomResource:(NSString *)url resouceType:(TEduBoardPreloadResourceType)resouceType expireTime:(NSInteger)expireTime callback:(nullable TEduResourceLoadCallback)callBack;
@end

NS_ASSUME_NONNULL_END
