//
//  TEduBoardDef.h
//  TEduBoard
//
//  Created by kennethmiao on 2019/4/2.
//  Copyright © 2019年 kennethmiao. All rights reserved.
//

#import <Foundation/Foundation.h>

#if TARGET_OS_IPHONE
#import <UIKit/UIKit.h>
typedef UIView TEView;
typedef UIColor TEColor;
typedef UIEdgeInsets TEEdgeInsets;
typedef UIImage TEImage;
#else
#import <AppKit/AppKit.h>
typedef NSView TEView;
typedef NSColor TEColor;
typedef NSEdgeInsets TEEdgeInsets;
typedef NSImage TEImage;
#endif

/**
 * @defgroup ctrl TEduBoardController
 * 白板功能的主要接口类
 */

/**
 * @defgroup callback TEduBoardDelegate
 * 白板功能的回调接口类
 */

/**
 * @defgroup def 关键类型定义
 */

/**
 * @defgroup code 错误码
 */


/**
 * @ingroup def
 * @brief 白板工具
*/
typedef NS_ENUM(NSInteger, TEduBoardToolType)
{
    TEDU_BOARD_TOOL_TYPE_MOUSE              = 0,    ///< 鼠标
    TEDU_BOARD_TOOL_TYPE_PEN                = 1,    ///< 画笔
    TEDU_BOARD_TOOL_TYPE_ERASER             = 2,    ///< 橡皮擦
    TEDU_BOARD_TOOL_TYPE_LASER              = 3,    ///< 激光笔
    TEDU_BOARD_TOOL_TYPE_LINE               = 4,    ///< 直线
    TEDU_BOARD_TOOL_TYPE_OVAL               = 5,    ///< 空心椭圆
    TEDU_BOARD_TOOL_TYPE_RECT               = 6,    ///< 空心矩形
    TEDU_BOARD_TOOL_TYPE_OVAL_SOLID         = 7,    ///< 实心椭圆 已废弃，请使用setGraphStyle接口来设置
    TEDU_BOARD_TOOL_TYPE_RECT_SOLID         = 8,    ///< 实心矩形 已废弃，请使用setGraphStyle接口来设置
    TEDU_BOARD_TOOL_TYPE_POINT_SELECT       = 9,    ///< 点选工具 2.6.2起废弃，请使用框选工具
    TEDU_BOARD_TOOL_TYPE_RECT_SELECT        = 10,   ///< 选框工具
    TEDU_BOARD_TOOL_TYPE_TEXT               = 11,   ///< 文本工具
    TEDU_BOARD_TOOL_TYPE_ZOOM_DRAG          = 12,   ///< 缩放移动
    TEDU_BOARD_TOOL_TYPE_SQUARE             = 13,   ///<空心正方形
    TEDU_BOARD_TOOL_TYPE_SQUARE_SOLID       = 14,   ///<实心正方形 已废弃，请使用setGraphStyle接口来设置
    TEDU_BOARD_TOOL_TYPE_CIRCLE             = 15,   ///<空心正圆形
    TEDU_BOARD_TOOL_TYPE_CIRCLE_SOLID       = 16,   ///<实心正圆形 已废弃，请使用setGraphStyle接口来设置
    TEDU_BOARD_TOOL_TYPE_BOARD_CUSTOM_GRAPH = 17,   ///<自定义图形，请配合addElement(TEDU_BOARD_ELEMENT_CUSTOM_GRAPH, '自定义图形URL')接口使用
    TEDU_BOARD_TOOL_TYPE_ARC                = 18,   ///<圆弧
    TEDU_BOARD_TOOL_TYPE_HIGHLIGHTER        = 19,   ///<荧光笔
    TEDU_BOARD_TOOL_TYPE_RIGHT_TRIANGLE     = 20,   ///<直角三角形
    TEDU_BOARD_TOOL_TYPE_ISOSCELES_TRIANGLE = 21,   ///<等腰三角形
    TEDU_BOARD_TOOL_TYPE_PARALLELOGRAM      = 22,   ///<平行四边形
    TEDU_BOARD_TOOL_TYPE_CUBE               = 23,   ///<立方体
    TEDU_BOARD_TOOL_TYPE_CYLINDER           = 24,   ///<圆柱
    TEDU_BOARD_TOOL_TYPE_CONE               = 25,   ///<圆锥
    TEDU_BOARD_TOOL_TYPE_COORDINATE         = 26,   ///<坐标系 
};

/**
 * @ingroup def
 * @brief 白板图片填充对齐模式
 * 当以宽度基准等比例放大，则居左和居右同居中对齐效果一致；当以高度基准等比例放大，则居顶和居底同居中对齐效果一致
 */
typedef NS_ENUM(NSInteger, TEduBoardImageFitMode) {
    TEDU_BOARD_IMAGE_FIT_MODE_CENTER        = 0,    ///< 以宽度或者高度为基准居中对齐等比例放大
    TEDU_BOARD_IMAGE_FIT_MODE_LEFT          = 4,    ///< 以宽度或者高度为基准左对齐等比例放大
    TEDU_BOARD_IMAGE_FIT_MODE_TOP           = 5,    ///< 以宽度或者高度为基准顶对齐等比例放大
    TEDU_BOARD_IMAGE_FIT_MODE_RIGHT         = 6,    ///< 以宽度或者高度为基准右对齐等比例放大
    TEDU_BOARD_IMAGE_FIT_MODE_BOTTOM        = 7,    ///< 以宽度或者高度为基准底对齐等比例放大
};

/**
 * @ingroup def
 * @brief 白板内容自适应模式
 * 内容包括图片、文件、PPT 动画
 */
typedef NS_ENUM(NSInteger, TEduBoardContentFitMode) {
    TEDU_BOARD_CONTENT_FIT_MODE_NONE            = 0,        ///< 不使用内容自适应，默认模式，不自动调整白板宽高比，内容等比例缩放居中显示，内容宽高 <= 白板宽高
    TEDU_BOARD_CONTENT_FIT_MODE_CENTER_INSIDE   = 1,        ///< 自动调整白板宽高比与内容一致，内容铺满白板，白板等比例缩放居中显示，白板宽高 <= 容器宽高
    TEDU_BOARD_CONTENT_FIT_MODE_CENTER_COVER    = 2,        ///< 自动调整白板宽高比与内容一致，内容铺满白板，白板等比例缩放居中显示，白板宽高 >= 容器宽高
};

/**
 * @ingroup def
 * @brief 白板图片状态
*/
typedef NS_ENUM(NSInteger, TEduBoardImageStatus)
{
    TEDU_BOARD_IMAGE_STATUS_LOADING         = 1,    ///< 背景图片正在加载
    TEDU_BOARD_IMAGE_STATUS_LOAD_DONE       = 2,    ///< 背景图片加载完成
    TEDU_BOARD_IMAGE_STATUS_LOAD_ABORT      = 3,    ///< 背景图片加载中断
    TEDU_BOARD_IMAGE_STATUS_LOAD_ERROR      = 4,    ///< 背景图片加载错误
    TEDU_BOARD_IMAGE_STATUS_LOAD_TIMEOUT    = 5,    ///< 背景图片加载超时
    TEDU_BOARD_IMAGE_STATUS_LOAD_CANCEL     = 6,    ///< 背景图片取消加载
    TEDU_BOARD_IMAGE_STATUS_READ_ERROR      = 7,    ///< 背景图片读取失败
};

/**
 * @ingroup def
 * @brief 白板文本样式
*/
typedef NS_ENUM(NSInteger, TEduBoardTextStyle)
{
    TEDU_BOARD_TEXT_STYLE_NORMAL            = 0,    ///< 常规样式
    TEDU_BOARD_TEXT_STYLE_BOLD              = 1,    ///< 粗体样式
    TEDU_BOARD_TEXT_STYLE_ITALIC            = 2,    ///< 斜体样式
    TEDU_BOARD_TEXT_STYLE_BOLD_ITALIC       = 3,    ///< 粗斜体样式
};

/**
 * @ingroup code
 * @brief 白板错误码（严重）
*/
typedef NS_ENUM(NSInteger, TEduBoardErrorCode)
{
    TEDU_BOARD_ERROR_INIT               = 1,       ///< 白板初始化失败
    TEDU_BOARD_ERROR_AUTH               = 2,       ///< 服务鉴权失败，未购买服务或服务已过期
    TEDU_BOARD_ERROR_LOAD               = 3,       ///< 白板加载失败
    TEDU_BOARD_ERROR_TIM_INVALID        = 5,       ///< 腾讯云 IMSDK 不可用
    TEDU_BOARD_ERROR_TIM_HISTORYDATA    = 6,       ///< 同步历史数据失败
    TEDU_BOARD_ERROR_RUNTIME            = 7,       ///< 白板运行错误
    TEDU_BOARD_ERROR_AUTH_TIMEOUT       = 8,       ///< 服务鉴权超时，有可能存在网络问题，请重试
    TEDU_BOARD_ERROR_DATA_TOO_LARGE     = 9,       ///< 传输的数据太大，请分拆成两个或多个后传递(比如，添加图片组文件时url数量太大)
    TEDU_BOARD_MAX_BOARD_LIMITED        = 10,      ///< 单课堂内白板页数已经到达上限
    TEDU_BOARD_SIGNATURE_EXPIRED        = 11,      ///< userSig过期, 请生成新的usersig，并重新初始化白板
    TEDU_BOARD_ERROR_PATH_INVALID       = 102,      ///< 路径非法
    TEDU_BOARD_ERROR_WRITE_ERROR        = 103,      ///< 文件写入错误
};

/**
 * @ingroup code
 * @brief 白板错误码（警告）
*/
typedef NS_ENUM(NSInteger, TEduBoardWarningCode)
{
    TEDU_BOARD_WARNING_SYNC_DATA_PARSE_FAILED           = 1,    ///< 接收到其他端的同步数据解析失败
    TEDU_BOARD_WARNING_TIM_SEND_MSG_FAILED              = 2,    ///< 腾讯 IMSDK 发送消息失败
    TEDU_BOARD_WARNING_H5PPT_ALREADY_EXISTS             = 3,    ///< 当要添加的 H5PPT 已存在时抛出该警告
    TEDU_BOARD_WARNING_ILLEGAL_OPERATION                = 5,    ///< 白板历史数据加载完成之前禁止操作
    TEDU_BOARD_WARNING_H5FILE_ALREADY_EXISTS            = 6,    ///< 当要添加的 H5File 已存在时抛出该警告
    TEDU_BOARD_WARNING_VIDEO_ALREADY_EXISTS             = 7,    ///< 当要添加的视频已存在时抛出该警告
    TEDU_BOARD_WARNING_IMAGESFILE_ALREADY_EXISTS        = 8,    ///< 当要添加的图片文件已存在时抛出该警告
    TEDU_BOARD_WARNING_GRAFFITI_LOST                    = 9,    ///< 有涂鸦丢失
    TEDU_BOARD_WARNING_CUSTOM_GRAPH_URL_NON_EXISTS      = 10,   ///< 自定义图形的URL不存在
    TEDU_BOARD_WARNING_IMAGESFILE_TOO_LARGE             = 11,   ///< addImagesFile 接口添加图片集合文件时，文件内容过大
    TEDU_BOARD_WARNING_IMAGE_COURSEWARE_ALREADY_EXISTS  = 12,   ///< addTranscodeFile 接口添加静态转码课件时，要添加的静态转码课件已存在
    TEDU_BOARD_WARNING_IMAGE_MEDIA_BITRATE_TOO_LARGE    = 13,   ///< 多媒体资源码率过大
    TEDU_BOARD_WARNING_IMAGE_WATERMARK_ALREADY_EXISTS   = 14,   ///< 水印已经存在
    TEDU_BOARD_WARNING_FORMULA_LIB_NOT_LOADED           = 15,   ///< 数学公式没有加载成功
    TEDU_BOARD_WARNING_ILLEGAL_FORMULA_EXPRESSION       = 16,   ///< 非法的数学公式表达时
    TEDU_BOARD_WARNING_TEXT_WATERMARK_ALREADY_EXISTS    = 17,   ///< 文本水印已经存在
    TEDU_BOARD_WARNING_EXPORTIMPORT_FILTERRULE_ILLEGAL  = 18,   ///< 导入导出过滤规则非法
    TEDU_BOARD_WARNING_ELEMENTTYPE_NOT_EXISTS           = 19,   ///< 元素类型不存在
    TEDU_BOARD_WARNING_ELEMENTID_NOT_EXISTS             = 20,   ///< 元素ID不存在
    TEDU_BOARD_WARNING_ELEMENT_IS_LOCKED                = 21,   ///< 元素当前是锁定状态，不允许操作
    TEDU_BOARD_WARNING_FILE_NOT_FOUND                   = 22,   ///< 文件没有找到，请重新添加文件
};

typedef NS_ENUM(NSInteger, TEduBoardTextComponentStatus)
{
    TEDU_BOARD_TEXT_COMPONENT_STATUS_INSUFFICIENT_SPACE       = 1,    ///< 空间不足，文本框无法显示
    TEDU_BOARD_TEXT_COMPONENT_STATUS_WORD_LIMIT          = 2,    ///< 文本超出字数限制
};
/**
 * @ingroup def
 * @brief 白板上传状态
*/
typedef NS_ENUM(NSInteger, TEduBoardUploadStatus)
{
    TEDU_BOARD_UPLOAD_STATUS_SUCCEED       = 1,    ///< 上传成功
    TEDU_BOARD_UPLOAD_STATUS_FAILED        = 2,    ///< 上传失败
};

/**
 * @ingroup def
 * @brief H5 背景状态
*/
typedef NS_ENUM(NSInteger, TEduBoardBackgroundH5Status)
{
    TEDU_BOARD_BACKGROUND_H5_STATUS_LOADING       = 1,    ///< H5 背景正在加载
    TEDU_BOARD_BACKGROUND_H5_STATUS_LOAD_FINISH   = 2,    ///< H5 背景加载完成
};

/**
 * @ingroup def
 * @brief 直线类型
*/
typedef NS_ENUM(NSInteger, TEduBoardLineType)
{
    TEDU_BOARD_LINE_TYPE_SOLID    = 1,    ///< 实线
    TEDU_BOARD_LINE_TYPE_DOTTED   = 2,    ///< 虚线
};

/**
 * @ingroup def
 * @brief 箭头类型
*/
typedef NS_ENUM(NSInteger, TEduBoardArrowType)
{
    TEDU_BOARD_ARROW_TYPE_NONE     = 1,    ///< 无箭头
    TEDU_BOARD_ARROW_TYPE_NORMAL   = 2,    ///< 普通箭头
    TEDU_BOARD_ARROW_TYPE_SOLID    = 3,    ///< 实心箭头
};

/**
 * @ingroup def
 * @brief 椭圆绘制模式
*/
typedef NS_ENUM(NSInteger, TEduBoardOvalDrawMode)
{
    TEDU_BOARD_OVAL_DRAW_MODE_FIX_START     = 1,    ///< 固定起始点，起始点和终点的中点为圆心
    TEDU_BOARD_OVAL_DRAW_MODE_FIX_CENTER    = 2,    ///< 固定圆心，起始点为圆心
};

/**
 * @ingroup def
 * @brief 文件转码状态
*/
typedef NS_ENUM(NSInteger, TEduBoardFileTranscodeStatus)
{
    TEDU_BOARD_FILE_TRANSCODE_ERROR         = 1,    ///< 转码错误
    TEDU_BOARD_FILE_TRANSCODE_UPLOADING     = 2,    ///< 文件上传中
    TEDU_BOARD_FILE_TRANSCODE_CREATED       = 3,    ///< 发起转码任务
    TEDU_BOARD_FILE_TRANSCODE_QUEUED        = 4,    ///< 排队中
    TEDU_BOARD_FILE_TRANSCODE_PROCESSING    = 5,    ///< 转码中
    TEDU_BOARD_FILE_TRANSCODE_FINISHED      = 6,    ///< 转码完成
};

/**
 * @ingroup def
 * @brief 视频文件状态
*/
typedef NS_ENUM(NSInteger, TEduBoardVideoStatus)
{
    TEDU_BOARD_VIDEO_STATUS_ERROR       = 1,    ///< 播放错误
    TEDU_BOARD_VIDEO_STATUS_LOADING     = 2,    ///< 加载中
    TEDU_BOARD_VIDEO_STATUS_LOADED      = 3,    ///< 加载完成
    TEDU_BOARD_VIDEO_STATUS_PLAYED      = 4,    ///< 开始播放
    TEDU_BOARD_VIDEO_STATUS_TIMEUPDATE  = 5,    ///< 时间更新
    TEDU_BOARD_VIDEO_STATUS_PAUSED      = 6,    ///< 暂停
    TEDU_BOARD_VIDEO_STATUS_SEEKED      = 7,    ///< 跳转
    TEDU_BOARD_VIDEO_STATUS_ENDED       = 8,    ///< 结束
    TEDU_BOARD_VIDEO_STATUS_WAITING     = 9,    ///< 缓冲中
    TEDU_BOARD_VIDEO_STATUS_PLAYING     = 10,    ///< 缓冲结束
};

/**
 * @ingroup def
 * @brief 音频状态
*/
typedef NS_ENUM(NSInteger, TEduBoardAudioStatus)
{
    TEDU_BOARD_AUDIO_STATUS_ERROR       = 1,    ///< 播放错误
    TEDU_BOARD_AUDIO_STATUS_LOADING     = 2,    ///< 加载中
    TEDU_BOARD_AUDIO_STATUS_LOADED      = 3,    ///< 加载完成
    TEDU_BOARD_AUDIO_STATUS_PLAYED      = 4,    ///< 开始播放
    TEDU_BOARD_AUDIO_STATUS_TIMEUPDATE  = 5,    ///< 时间更新
    TEDU_BOARD_AUDIO_STATUS_PAUSED      = 6,    ///< 暂停
    TEDU_BOARD_AUDIO_STATUS_SEEKED      = 7,    ///< 跳转
    TEDU_BOARD_AUDIO_STATUS_ENDED       = 8,    ///< 结束
    TEDU_BOARD_AUDIO_STATUS_WAITING     = 9,    ///< 缓冲中
    TEDU_BOARD_AUDIO_STATUS_PLAYING     = 10,    ///< 缓冲结束
};


/**
 * @ingroup def
 * @brief PPT 动画文件加载状态
*/
typedef NS_ENUM(NSInteger, TEduBoardH5PPTStatus)
{
    TEDU_BOARD_H5_PPT_STATUS_LOADING                            = 1,    ///< PPT动画文件加载中
    TEDU_BOARD_H5_PPT_STATUS_LOADED                             = 2,    ///< PPT动画文件加载完成
    TEDU_BOARD_H5_PPT_STATUS_CANCEL                             = 3,    ///< PPT动画文件取消加载 (本地PPT还没有加载完成，就切换到了其他文件)
    TEDU_BOARD_H5_PPT_STATUS_TIMEOUT                            = 4,    ///< PPT动画文件加载超时
    TEDU_BOARD_H5_PPT_STATUS_RESOURCES_LOADING_ERROR            = 5,    ///< PPT动画文件加载失败
    TEDU_BOARD_H5_PPT_STATUS_RUNTIME_ERROR                      = 6,    ///< PPT动画文件加载内部运行错误
    TEDU_BOARD_H5_PPT_STATUS_ADD_TIMEOUT                        = 7,    ///< 调用addTranscodeFile接口添加文件的时候超时，监听到此错误，可重新调用addTranscodeFile进行重试添加
    TEDU_BOARD_H5_PPT_STATUS_IMAGE_ABNORMAL                     = 8,    ///< ppt内部的图片资源加载异常，监听到此状态码，可配合 refresh接口重新渲染白
    TEDU_BOARD_H5_PPT_STATUS_INTERNAL_RESOURCES_LOADING_TIMEOUT = 9,///< ppt内部资源加载超时
};

/**
 * @ingroup def
 * @brief PPT 媒体状态
*/
typedef NS_ENUM(NSInteger, TEduBoardH5PPTMediaStatus)
{
    TEDU_BOARD_H5_PPT_MEDIA_STATUS_PLAY                    = 1,    ///< PPT媒体播放
    TEDU_BOARD_H5_PPT_MEDIA_STATUS_PAUSE                   = 2,    ///< PPT媒体暂停
};

/**
 * @ingroup def
 * @brief H5 文件状态
*/
typedef NS_ENUM(NSInteger, TEduBoardH5FileStatus)
{
    TEDU_BOARD_H5_FILE_STATUS_LOADING       = 1,    ///< 加载中
    TEDU_BOARD_H5_FILE_STATUS_LOADED        = 2,    ///< 加载完成
};

/**
 * @ingroup def
 * @brief 元素类型
*/
typedef NS_ENUM(NSInteger, TEduBoardElementType)
{
    TEDU_BOARD_ELEMENT_IMAGE                                    =   1,      ///<图片元素
    TEDU_BOARD_ELEMENT_H5                                       =   2,      ///<网页元素
    TEDU_BOARD_ELEMENT_CUSTOM_GRAPH                             =   3,      ///<自定义图形，请与自定义图形工具TEDU_BOARD_TOOL_TYPE_BOARD_CUSTOM_GRAPH配合使用
    TEDU_BOARD_ELEMENT_AUDIO                                    =   4,      ///<音频元素，当前白板有效
    TEDU_BOARD_ELEMENT_GLOBAL_AUDIO                             =   5,      ///<全局音频元素，无控制界面，一般为背景音乐
    TEDU_BOARD_ELEMENT_MATH_BOARD                               =   6,      ///<数学函数画板
    TEDU_BOARD_ELEMENT_MATH_GRAPH                               =   7,      ///<数学函数图像
    TEDU_BOARD_ELEMENT_TEXT                                     =   9,      ///<文本元素
    TEDU_BOARD_ELEMENT_MAGIC_LINE                               =   10,     ///<魔法笔元素
    TEDU_BOARD_ELEMENT_FORMULA                                  =   11,     ///<数学公式元素
    TEDU_BOARD_ELEMENT_MATH_TOOL                                =   12,     ///<数学教具
    TEDU_BOARD_ELEMENT_WATERMARK                                =   100,     ///<水印
    TEDU_BOARD_ELEMENT_GRAFFITI_LINE                            =   801,    ///<画线涂鸦
    TEDU_BOARD_ELEMENT_GRAFFITI_GRAPH_LINE                      =   802,    ///<直线涂鸦
    TEDU_BOARD_ELEMENT_GRAFFITI_GRAPH_RECT                      =   803,    ///<矩形涂鸦
    TEDU_BOARD_ELEMENT_GRAFFITI_GRAPH_OVAL                      =   804,    ///<椭圆涂鸦
    TEDU_BOARD_ELEMENT_GRAFFITI_GRAPH_ARC                       =   805,    ///<圆弧元素
    TEDU_BOARD_ELEMENT_GRAFFITI_GRAPH_RIGHT_TRIANGLE            =   806,    ///<直角三角形
    TEDU_BOARD_ELEMENT_GRAFFITI_GRAPH_ISOSCELES_TRIANGLE        =   807,    ///<等腰三角形
    TEDU_BOARD_ELEMENT_GRAFFITI_GRAPH_PARALLELOGRAM             =   808,    ///<平行四边形
    TEDU_BOARD_ELEMENT_GRAFFITI_GRAPH_CUBE                      =   809,    ///<立方体
    TEDU_BOARD_ELEMENT_GRAFFITI_GRAPH_CYLINDER                  =   810,    ///<圆柱体
    TEDU_BOARD_ELEMENT_GRAFFITI_GRAPH_CONE                      =   811,    ///<圆锥体
    TEDU_BOARD_ELEMENT_GRAFFITI_GRAPH_HIGHLIGHTER               =   812,    ///<荧光笔
};

/**
 * @ingroup def
 * @brief 位置类型
*/
typedef NS_ENUM(NSInteger, TEduBoardPosition)
{
    TEDU_BOARD_POSITION_LEFT_TOP    =  0,       ///< 左上方
    TEDU_BOARD_POSITION_RIGHT_TOP    =  1,      ///< 右上方
    TEDU_BOARD_POSITION_RIGHT_BOTTOM    =  2,   ///< 右下方
    TEDU_BOARD_POSITION_LEFT_BOTTOM    =  3,    ///< 左下方
};

/**
 * @ingroup def
 * @brief 位置类型
*/
typedef NS_ENUM(NSInteger, TEduBoardAccessibleOperation)
{
    TEDU_BOARD_OPERATION_ALL        =  0,       ///< 全部操作
    TEDU_BOARD_OPERATION_MOVE       =  1,       ///< 移动
    TEDU_BOARD_OPERATION_DELETE     =  2,       ///< 删除
    TEDU_BOARD_OPERATION_SCALE      =  3,       ///< 缩放
    TEDU_BOARD_OPERATION_ROTATE     =  4,       ///< 旋转
    TEDU_BOARD_OPERATION_SELECTED   =  5,       ///< 选中
    TEDU_BOARD_OPERATION_ADD        =  6,       ///< 添加
};

/**
 * @ingroup def
 * @brief 数学函数图像工具状态码
*/
typedef NS_ENUM(NSInteger, TEduBoardMathGraphCode)
{
    MATH_EXPRESSION_PARSE_ERROR        =        -1,       ///< 函数表达式解析失败
};

/**
 * @ingroup def
 * @brief 画笔自动拟合模式
*/
typedef NS_ENUM(NSInteger, TEduBoardPenFittingMode)
{
    TEduBoardPenFittingModeNone             =        0,       ///< 不进行曲线拟合
    TEduBoardPenFittingModeAuto             =        1,       ///< 自动识别并拟合图形
    TEduBoardPenFittingModeRectangle        =        2,       ///< 拟合矩形
    TEduBoardPenFittingModePolygon          =        3,       ///< 拟合多边形
    TEduBoardPenFittingModeCircle           =        4,       ///< 拟合圆形
    TEduBoardPenFittingModeEllipse          =        5,       ///< 拟合椭圆
};

/**
 * @ingroup def
 * @brief 白板背景类型
*/
typedef NS_ENUM(NSInteger, TEduBoardBackgroundType)
{
    TEDU_BOARD_BACKGROUND_IMAGE             =        1,       ///< 图片类型
    TEDU_BOARD_BACKGROUND_H5                =        2,       ///< 网页类型
};


/**
 * @ingroup def
 * @brief 白板背景类型
*/
typedef NS_ENUM(NSInteger, TEduBoardMathGraphType)
{
    TEduBoardMathGraphTypeNone              =        0,       ///< 无图形输入，此时可选中几何图形
    TEduBoardMathGraphTypePoint             =        1,       ///< 点
    TEduBoardMathGraphTypeLine              =        2,       ///< 直线
    TEduBoardMathGraphTypeLineSegment       =        3,       ///< 线段
    TEduBoardMathGraphTypeRay               =        4,       ///< 射线
    TEduBoardMathGraphTypeCircle            =        5,       ///< 圆
    TEduBoardMathGraphTypeAngle             =        6,       ///< 角
    TEduBoardMathGraphTypePolygon           =        7,       ///< 多边形
    TEduBoardMathGraphTypeVector            =        8,       ///< 向量
    TEduBoardMathGraphTypeEllipse           =        9,       ///< 椭圆
    TEduBoardMathGraphTypeCube              =        101,       ///< 立方体
    TEduBoardMathGraphTypeCylinder          =        102,       ///< 圆柱体
    TEduBoardMathGraphTypeCircularCone      =        103,       ///< 圆锥体
};

/**
 * @ingroup def
 * @brief 白板分组操作
*/
typedef NS_ENUM(NSInteger, TEduBoardClassGroupOperationType)
{
    TEduBoardClassGroupOperationTypeSetEnable              =        0,       ///< 开启分组模式
    TEduBoardClassGroupOperationTypeUpdate                 =        1,       ///< 更新分组信息
    TEduBoardClassGroupOperationTypeDelete                 =        2,       ///< 删除分组
    TEduBoardClassGroupOperationTypeReset                  =        3,       ///< 重置分组
    TEduBoardClassGroupOperationTypeError                  =        4,       ///< 分组功能发生错误
};

/**
 * @ingroup def
 * @brief 白板可擦除元素类型
*/
typedef NS_ENUM(NSInteger, TEduBoardErasableElementType)
{
    TEduBoardErasableElementTypeLine                =        0,       ///< 画线元素
    TEduBoardErasableElementTypeGraph               =        1,       ///< 图形元素
    TEduBoardErasableElementTypeText                =        2,       ///< 文本元素
    TEduBoardErasableElementTypeImage               =        3,       ///< 图片元素
    TEduBoardErasableElementTypeElement             =        4,       ///< 自定义元素
};

/**
 * @ingroup def
 * @brief 白板可擦除元素类型
*/
typedef NS_ENUM(NSInteger, TEduBoardLogLevel)
{
    TEduBoardLogLevelAll                    =        0,       ///< 全部日志
    TEduBoardLogLevelInfo                   =        1,       ///< 一般日志
    TEduBoardLogLevelWarning                =        2,       ///< 告警日志
    TEduBoardLogLevelError                  =        3,       ///< 错误日志
    TEduBoardLogLevelOff                    =        4,       ///< 不打印日志
};

/**
 * @ingroup def
 * @brief 白板文件类型
*/
typedef NS_ENUM(NSInteger, TEduBoardFileType)
{
    TEduBoardFileTypeWhiteBoard                 =        1,       ///< 默认白板组
    TEduBoardFileTypeH5Courseware               =        2,       ///< 动态转码课件
    TEduBoardFileTypeImageCourseware            =        3,       ///< 静态转码课件
    TEduBoardFileTypeVideoFile                  =        4,       ///< 视频文件
    TEduBoardFileTypeH5File                     =        5,       ///< H5文件
    TEduBoardFileTypeIMagesFile                 =        6,       ///< 图片组文件
};

/**
 * @ingroup def
 * @brief 白板文件类型
*/
typedef NS_ENUM(NSInteger, TEduBoardRemoteCursorAction)
{
    TEduBoardRemoteCursorMoving                =        0,       ///< 正在移动
    TEduBoardRemoteCursorDrawing               =        1,       ///< 正在绘制
};


/**
 * @ingroup def
 * @brief 数学教具
*/
typedef NS_ENUM(NSInteger, TEduMathToolType)
{
    TEduMathToolTypeRuler                          =        1,       ///< 直尺
    TEduMathToolTypeTriangle                       =        2,       ///< 直角三角形（30°，60°，90°）
    TEduMathToolTypeIsoscelesTriangle              =        3,       ///< 等腰直角三角形（45°，45°，90°）
    TEduMathToolTypeProtractor                     =        4,       ///< 量角器
    TEduMathToolTypeCompasses                      =        5,       ///< 圆规
};

/**
 * @ingroup def
 * @brief 白板图形填充模式
*/
typedef NS_ENUM(NSInteger, TEduBoardFillType)
{
    TEduBoardFillTypeNone                         =        1,       ///< 不填充
    TEduBoardFillTypeSolid                        =        2,       ///< 填充
};

typedef NS_ENUM(NSInteger, TEduElementOperatorType)
{
    TEduElementOperatorTypeBorderColor            =        1,       ///< border颜色
    TEduElementOperatorTypeBorderWidth            =        2,       ///< border宽度
    TEduElementOperatorTypeBorderLineType         =        3,       ///< 实线还是虚线
    TEduElementOperatorTypeElementDelete          =        4,       ///< 删除元素
    TEduElementOperatorTypePosition               =        5,       ///< 位置
    TEduElementOperatorTypeTextSize               =        6,       ///< 修改文本元素字体大小
    TEduElementOperatorTypeTextColor              =        7,       ///< 修改文本元素文本颜色
    TEduElementOperatorTypeContent                =        8,       ///< 改变内容
    TEduElementOperatorTypeOpacity                =        9,       ///< 改变透明度
    TEduElementOperatorTypeRotationAngle          =        10,       ///< 改变透明度

};

/**
 * @ingroup def
 * @brief 导入数据状态
*/
typedef NS_ENUM(NSInteger, TEduBoardImportInLocalModeStatus)
{
    TEduBoardImportInLocalModeStatusSuccess                        =        0,       ///< 导入成功
    TEduBoardImportInLocalModeStatusParseDataError                 =        1,       ///< 数据解析异常
};

/**
 * @ingroup def
 * @brief 水印类型
*/
typedef NSString *TEduWatermarkType NS_STRING_ENUM;
FOUNDATION_EXPORT TEduWatermarkType const TEduWatermarkTypeImage;
FOUNDATION_EXPORT TEduWatermarkType const TEduWatermarkTypeText;
FOUNDATION_EXPORT TEduWatermarkType const TEduWatermarkTypeTextFix;

/**
 * @ingroup def
 * @brief 导入数据状态
*/
typedef NS_ENUM(NSInteger, TEduBoardPositionChangeStatus)
{
    TEDU_BOARD_POSITION_CHANGE_START             =        0,       ///< 开始变化
    TEDU_BOARD_POSITION_CHANGING                 =        1,       ///< 变化中
    TEDU_BOARD_POSITION_CHANGE_END               =        2,       ///< 结束变化
};


/**
 * @ingroup def
 * @brief 截图功能的状态码
*/
typedef NS_ENUM(NSInteger, TEduSnapshotCode)
{
    TEDU_SNAPSHOT_NONE                                      =        0,       ///< 截图正常
    TEDU_SNAPSHOT_IMAGE_LOAD_FAILED                         =        -1,       ///< 图片加载失败，不影响截图功能，只是图片没有被正确截取
    TEDU_SNAPSHOT_IMAGE_DIMENSIONS_ABNORMAL                 =        -100,       ///< 图片尺寸获取异常
};


/**
 * @ingroup def
 * @brief 绘画动作状态切换
*/
typedef NS_ENUM(NSInteger, TEduBoardDrawStatusChanged)
{
    TEduBoardDrawStatusStart             =        1,        ///< 绘制开始
    TEduBoardDrawStatusEnd               =        0,        ///< 绘制结束
};
/**
 * @ingroup def
 * @brief 白板信息
*/
@interface TEduBoardUserInfo : NSObject
@property (nonatomic, strong) NSString *nickname;        ///< 用户昵称
@end

/**
 * @ingroup def
 * @brief 白板信息
*/
@interface TEduBoardInfo : NSObject
@property (nonatomic, strong) NSString *boardId;        ///< 白板 ID
@property (nonatomic, strong) NSString *backgroundUrl;  ///< 背景图或者背景 H5 页面的 URL
@property (nonatomic, strong) TEColor *backgroundColor; ///< 白板背景色
@end

/**
 * @ingroup def
 * @brief 课件媒体信息
*/

@interface TEduBoardPPTMediaInfo : NSObject

@property (nonatomic, strong) NSArray<NSString *> *videoIDList;
@property (nonatomic, strong) NSArray<NSString *> *audioIDList;

@end

/**
 * @ingroup def
 * @brief 文件信息
*/
@interface TEduBoardFileInfo : NSObject

@property (nonatomic, strong) NSString *fileId;                         ///< 文件标识
@property (nonatomic, strong) NSString *title;                          ///< 文件名称
@property (nonatomic, strong) NSString *downloadURL;                    ///< 文件下载地址
@property (nonatomic, assign) int pageIndex;                            ///< 当前页数
@property (nonatomic, assign) int pageCount;                            ///< 页数
@property (nonatomic, strong) NSArray<TEduBoardInfo *> *boardInfoList;  ///< 白板信息列表
@property (nonatomic, assign) TEduBoardFileType fileType;               ///< 文件确切类型`TEduBoardFileType`
@property (nonatomic, strong) TEduBoardPPTMediaInfo *mediaInfo;         ///< ppt文件里的多媒体信息


@end

/**
 * @ingroup def
 * @brief 元素信息
*/
@interface TEduBoardElementInfo : NSObject
@property (nonatomic, strong) NSString *boardId;                        ///< 白板 ID
@property (nonatomic, strong) NSString *elementId;                      ///< 元素 ID
@property (nonatomic, assign) TEduBoardElementType type;                ///< 元素类型
@property (nonatomic, strong) NSString *url;                            ///< 元素地址
@property (nonatomic, assign) double left;
@property (nonatomic, assign) double top;
@property (nonatomic, assign) double width;
@property (nonatomic, assign) double height;
@property (nonatomic, assign) double rotate;
@property (nonatomic, assign) CGRect boundingbox;
@property (nonatomic, assign) BOOL show;
@end


@interface TEduBoardElementStyle : NSObject

@property (nonatomic, assign) double borderWidth;                           ///< 笔刷线宽
@property (nonatomic, strong) NSString *borderColor;                        ///< 笔刷颜色
@property (nonatomic, assign) BOOL fill;                                    ///< 是否填充
@property (nonatomic, strong) NSString *fillColor;                          ///< 填充颜色
@property (nonatomic, assign) TEduBoardLineType borderType;                 ///< 线类型
@property (nonatomic, assign) TEduBoardArrowType startArrowType;            ///< 开始箭头类型， 只有直线支持
@property (nonatomic, assign) TEduBoardArrowType endArrowType;              ///< 结束箭头类型， 只有直线支持
///
@end

/**
 * @ingroup def
 * @brief 元素信息
*/
@interface TEduBoardElement : NSObject

@property (nonatomic, strong) NSString *boardId;                        ///< 白板 ID
@property (nonatomic, strong) NSString *elementId;                      ///< 元素 ID
@property (nonatomic, assign) TEduBoardElementType type;                ///< 元素类型
@property (nonatomic, strong) NSString *userId;                         ///< 元素创建者
@property (nonatomic, strong) NSString *title;                          ///< 元素标题
@property (nonatomic, assign) BOOL show;                                ///< 元素是否显示
@property (nonatomic, assign) BOOL select;                              ///< 元素是否被选中
@property (nonatomic, assign) CGRect rectBox;                           ///< 元素原始包围盒（不旋转）
@property (nonatomic, assign) double rectBoxRotate;                     ///< 元素原始包围盒旋转度
@property (nonatomic, assign) CGRect boundingbox;                       ///< 元素旋转后包围盒
@property (nonatomic, assign) NSInteger textSize;                       ///< 文本字体大小；非文本元素为0
@property (nonatomic, strong) TEduBoardElementStyle *style;             ///< 元素样式
@property (nonatomic, strong) NSString *content;                        ///< 元素内容
@property (nonatomic, assign) CGFloat opacity;                          ///< 元素透明度
@property (nonatomic, assign) BOOL locked;                              ///< 是否锁定


@end

/**
 * @ingroup def
 * @brief 授权参数
*/
@interface TEduBoardAuthParam : NSObject
@property (nonatomic, assign) NSInteger sdkAppId;         ///< 应用标识
@property (nonatomic, strong) NSString *userId;     ///< 用户标识
@property (nonatomic, strong) NSString *userSig;    ///< 签名
@property (nonatomic, strong) NSString *imGroupId;  ///< IM群组ID，默认不传，当IM群组与白板房间号不一致时使用
@end

/**
 * @ingroup def
 * @brief 白板初始化常规配置参数
*/
@interface TEduBoardInitConfig : NSObject
@property (nonatomic, strong) NSString *ratio;                              ///<默认白板宽高比，可传格式如“4:3”、“16:9”的字符串，默认“16:9”
@property (nonatomic, assign) UInt32 scale;                                 ///<白板默认缩放系数，实际缩放倍数为 scale/100
@property (nonatomic, assign) TEduBoardToolType toolType;                   ///<白板工具，默认 TEDU_BOARD_TOOL_TYPE_PEN
@property (nonatomic, assign) TEduBoardContentFitMode boardContentFitMode;  ///<白板内容自适应模式，默认 TEDU_BOARD_CONTENT_FIT_MODE_NONE
@property (nonatomic, assign) NSInteger preloadDepth;                       ///<图片预加载深度，默认值5，表示预加载当前页前后5页的图片
@property (nonatomic, assign) NSInteger syncFps;                            ///<信令同步频率，该值的允许范围为[5, 20]，默认5
@property (nonatomic, strong) NSString *progressBarUrl;                     ///<自定义加载图标，在 progressEnable = true 时生效，支持 jpg、gif、png、svg
@property (nonatomic, assign) UInt32 autoHideRemoteCursorTimeout;           ///<远端隐藏画笔时间(毫秒)
@property (nonatomic, strong) NSArray<NSNumber*> *scaleRange;               ///<白板缩放范围，百分比形式，100表示100%，无缩放，默认值 @[@100, @1600]
@property (nonatomic, assign) NSInteger offlineWarningTimeout;              ///<白板离线告警时间间隔，单位（秒）
@property (nonatomic, assign) NSInteger h5PPTLoadTimeout;                   ///<H5 PPT加载超时时间，单位（秒），默认10s
@property (nonatomic, assign) NSInteger h5PPTResourceTimeout;               ///<H5 PPT内部资源加载超时时间，单位（秒），默认10s
///<  H5 PPT加载超时次数达到次数后，触发降级渲染通知；默认0，表示未设置，不触发降级通知；
///<  收到通知后需要调用 - (void)setDownGrade:(NSString *)fileId slideIndex:(NSInteger)slideIndex enable:(BOOL)enable;进行降级渲染
@property (nonatomic, assign) NSInteger h5PPTDownGradeTimeoutTimes;
@property (nonatomic, assign) BOOL videoFileDrawEnable;                                       ///<是否允许视频文件涂鸦，默认 NO
@end

/**
 * @ingroup def
 * @brief 用户配置参数
*/
@interface TEduBoardInitUserConfig : NSObject
@property (nonatomic, strong) NSString *nickname;                              ///<当前白板用户昵称
@end


/**
 * @ingroup def
 * @brief 背景图片信息
*/
@interface TEduBoardBackgroundImageInfo : NSObject
@property (nonatomic, strong) NSString* url;                        ///< 图片资源链接
@property (nonatomic, assign) TEduBoardImageFitMode mode;       ///< 图片填充对齐模式
@property (nonatomic, assign) NSInteger type;                       ///< 背景类型（0: 背景图 1：H5）
@property (nonatomic, assign) NSInteger angle;                      ///< 角度信息
@end


/**
 * @ingroup def
 * @brief 样式配置参数
*/
@interface TEduBoardInitStyleConfig : NSObject
@property (nonatomic, assign) TEduBoardTextStyle textStyle;                             ///<文本样式，默认 TEDU_BOARD_TEXT_STYLE_NORMAL
@property (nonatomic, assign) int textSize;                                             ///<文本粗细，默认320
@property (nonatomic, strong) TEColor *textColor;                                       ///<文本颜色
@property (nonatomic, strong) TEColor *brushColor;                                      ///<画笔颜色
@property (nonatomic, assign) int brushThin;                                            ///<画笔粗细，默认100
@property (nonatomic, strong) TEColor *globalBackgroundColor;                           ///<全局背景色
@property (nonatomic, strong) TEduBoardBackgroundImageInfo *globalBackgroundPic;      ///<全局背景图
@property (nonatomic, strong) TEColor *selectBoxColor;                                  ///<选框颜色，包括选框上的缩放，旋转按钮
@property (nonatomic, strong) TEColor *selectAnchorColor;                               ///<选框缩放控制点颜色
@property (nonatomic, strong) NSArray<NSNumber*> *cursorSize;                           ///<画笔大小限制
@property (nonatomic, strong) TEColor *scrollbarThumbColor;                             ///< 滚动条滑块颜色
@property (nonatomic, strong) TEColor *scrollbarTrackColor;                             ///< 滚动条滑轨颜色
@end


/**
 * @ingroup def
 * @brief 白板元素操作权限
*/
@interface TEduBoardElementOperationAuthority: NSObject
@property (nonatomic, assign) bool disableRotate;                   ///<是否禁用旋转 （文本元素默认为true，不可更改）
@property (nonatomic, assign) bool disableMove;                     ///<是否禁用移动
@property (nonatomic, assign) bool disableProportionScale;          ///<是否禁用等比例缩放
@property (nonatomic, assign) bool disableArbitraryScale;           ///<是否禁用任意缩放 （文本元素默认为true，不可更改）
@property (nonatomic, assign) bool disableUseEmoji;                 ///<是否禁用使用emoji表情 （文本元素默认为true，不可更改）
@end

/**
 * @ingroup def
 * @brief 白板操作权限
*/
@interface TEduBoardOperationAuthority: NSObject
@property (nonatomic, strong) TEduBoardElementOperationAuthority *line;             ///<画笔涂鸦元素
@property (nonatomic, strong) TEduBoardElementOperationAuthority *graph;            ///<图形涂鸦元素
@property (nonatomic, strong) TEduBoardElementOperationAuthority *text;             ///<文本元素
@property (nonatomic, strong) TEduBoardElementOperationAuthority *formula;          ///<公式
@property (nonatomic, strong) TEduBoardElementOperationAuthority *watermark;        ///<水印元素
@property (nonatomic, strong) TEduBoardElementOperationAuthority *customGraph;      ///<自定义图形
@property (nonatomic, strong) TEduBoardElementOperationAuthority *image;            ///<图片元素
@property (nonatomic, strong) TEduBoardElementOperationAuthority *h5;               ///<网页元素
@property (nonatomic, strong) TEduBoardElementOperationAuthority *mathTool;         ///<数学教具
@property (nonatomic, strong) TEduBoardElementOperationAuthority *audio;            ///<音频元素
@property (nonatomic, strong) TEduBoardElementOperationAuthority *mathBoard;        ///<数学画板
@property (nonatomic, strong) TEduBoardElementOperationAuthority *magicLine;        ///<魔法笔
@end

/**
 * @ingroup def
 * @brief 鼠标工具行为 -- 翻页配置参数
*/
@interface TEduBoardMouseToolTurnPage : NSObject
@property (nonatomic, assign) bool whiteBoard;               ///< 普通白板点击可翻页
@property (nonatomic, assign) bool h5PPT;                   ///< 动态ppt转码的文件点击可翻页
@property (nonatomic, assign) bool imgPPT;                  ///< 静态ppt转码文件点击不可翻页
@property (nonatomic, assign) bool imgFile;                 ///< 图片文件（addImagesFile接口添加的文件）点击不可翻页
@end

/**
 * @ingroup def
 * @brief 鼠标工具行为
*/
@interface TEduBoardMouseToolBehavior : NSObject
@property (nonatomic, strong)TEduBoardMouseToolTurnPage *turnPage;
@end

/**
 * @ingroup def
 * @brief 画笔模式下远端画笔是否显示
*/
@interface TEduBoardCursorVisible : NSObject
@property (nonatomic, assign)bool drawing;      ///< 绘制状态远端画笔是否显示, 默认为true
@property (nonatomic, assign)bool moving;       ///< 移动状态远端画笔是否显示, 默认为true
@end

/**
 * @ingroup def
 * @brief 权限配置参数
*/
@interface TEduBoardInitAuthConfig : NSObject
@property (nonatomic, assign) bool drawEnable;                                          ///<是否允许涂鸦，默认 YES
@property (nonatomic, assign) bool dataSyncEnable;                                      ///<是否启用数据同步，禁用后将导致本地白板操作不会被同步给远端
@property (nonatomic, assign) bool progressEnable;                                      ///<是否启用SDK内置Loading图标
@property (nonatomic, assign) bool systemCursorEnable;                                  ///<是否启用原生系统光标，默认false
@property (nonatomic, assign) bool enableScaleTool;                                     ///<是否启用白板缩放移动工具的缩放功能，默认为true
@property (nonatomic, assign) bool isAutoHideRemoteCursor;                              ///<远端画笔不动2秒后是否自动隐藏
@property (nonatomic, assign) bool showRemoteSelectBox;                                 ///<是否显示远端选框
@property (nonatomic, assign) bool showLocalOperator;                                   ///<是否显示本地操作信息（选框工具选中白板元素显示白板元素的创建者）
@property (nonatomic, assign) bool showRemoteOperator;                                  ///<是否显示远端操作信息（远端使用选框工具时，本地显示远端用户操作的白板元素）
@property (nonatomic, assign) bool showCursorOnTouch;                                   ///<是否在移动端上显示画笔
@property (nonatomic, assign) bool pinchToZoomEnable;                                   ///<在任意工具下开启双指缩放
@property (nonatomic, assign) bool mathGraphEnable;                                     ///<是否预加载数学函数图像库
@property (nonatomic, assign) bool formulaEnable;                                       ///<是否预加载公式库
@property (nonatomic, assign) bool customCursorIcon;                                    ///<是否自定义鼠标样式, 默认false； 设置为为true时， 可收到`[TEduBoardDelegate onTEBBoardCursorPosition]` 回调事件，自定义鼠标样式
@property (nonatomic, strong) TEduBoardCursorVisible *remoteCursorVisible;              ///<画笔模式下远端画笔是否显示
@property (nonatomic, strong) TEduBoardOperationAuthority *elementOperationAuthority;   ///<设置白板元素操作权限
@property (nonatomic, strong) TEduBoardMouseToolBehavior *mouseToolBehavior;            ///<设置鼠标工具行为
@end

/**
 * @ingroup def
 * @brief 自定义服务配置，在特定场景下使用，正常场景下不需要配置
*/
@interface TEduBoardCustomServerConfig : NSObject
@property (nonatomic, strong) NSString *serverHost;          ///<白板后台服务host，必填，需要带上协议 如 http://123.456.789.0:3456
@property (nonatomic, strong) NSString *serverPath;          ///<白板后台服务路径，选填，该参数会拼接在serverHost 如：/xxxx/tiw，最终请求为：http://123.456.789.0:3456/xxxx/tiw
@property (nonatomic, strong) NSString *resourceHost;        ///<白板SDK资源包部署服务的host，选填，需要带上协议，http://123.4.5.6:8081
@property (nonatomic, strong) NSString *token;               ///<jwt鉴权token，会携带在白板后台服务请求的header中，选填，如果业务侧需要对请求进行鉴权，可配置此参数
@end


/**
 * @ingroup def
 * @brief 白板初始化参数
*/

@interface TEduBoardInitParam : NSObject

@property (nonatomic, strong) TEduBoardInitConfig *config;                          ///<常规配置参数
@property (nonatomic, strong) TEduBoardInitUserConfig *userConfig;                  ///<用户配置参数
@property (nonatomic, strong) TEduBoardInitStyleConfig *styleConfig;                ///<样式配置参数
@property (nonatomic, strong) TEduBoardInitAuthConfig *authConfig;                  ///<权限配置参数
@property (nonatomic, strong) TEduBoardCustomServerConfig *customServerConfig;      ///<自定义服务配置，在特定场景下使用，正常场景下不需要配置

@property (nonatomic, assign) bool timSync;                     ///<是否使用腾讯云 IMSDK 进行实时数据同步，默认 YES
@property (nonatomic, assign) float smoothLevel;                ///<平滑系数，默认0，范围[0，1]
@property (nonatomic, assign) int imageTimeout;                 ///<图片加载超时时间，单位秒默认10s
@property (nonatomic, assign) BOOL disableInputAccessoryView;   ///<是否禁用输入框键盘工具栏，默认 NO
@property (nonatomic, assign) BOOL enableRequestReferer;        ///<是否启用请求携带referer，默认 NO。referer 值为：https://res.qcloudtiw.com 或 teducache://res.qcloudtiw.com

@end

/**
 * @ingroup def
 * @brief 直线样式
*/
@interface TEduBoardLineStyle : NSObject
@property (nonatomic, assign) TEduBoardLineType lineType;           ///<直线类型，默认 TEDU_BOARD_LINE_TYPE_SOLID
@property (nonatomic, assign) TEduBoardArrowType startArrowType;    ///<起点箭头类型，默认 TEDU_BOARD_ARROW_TYPE_NONE
@property (nonatomic, assign) TEduBoardArrowType endArrowType;      ///<终点箭头类型，默认 TEDU_BOARD_ARROW_TYPE_NONE
@end


/**
 * @ingroup def
 * @brief 图形样式
*/
@interface TEduBoardGraphStyle : NSObject
@property (nonatomic, assign) TEduBoardLineType lineType;           ///<直线类型，`TEduBoardLineType`
@property (nonatomic, assign) TEduBoardFillType fillType;           ///<填充类型, 即实心空心，只对平面几何图形有效，`TEduBoardFillType`
@property (nonatomic, assign) TEduBoardArrowType startArrowType;    ///<起点箭头类型 默认无箭头，只对直线有效，`TEduBoardArrowType`
@property (nonatomic, assign) TEduBoardArrowType endArrowType;    ///<终点箭头类型 默认无箭头，只对直线有效，`TEduBoardArrowType`

// 虚线直线，无箭头 需同时设置setToolType:TEDU_BOARD_TOOL_TYPE_LINE
+ (TEduBoardGraphStyle*)dashedLineNoArrow;
// 实线直线，终点实心箭头 需同时设置setToolType:TEDU_BOARD_TOOL_TYPE_LINE
+ (TEduBoardGraphStyle*)solidLineEndArrow;
// 虚线直线，起点普通箭头 需同时设置setToolType:TEDU_BOARD_TOOL_TYPE_LINE
+ (TEduBoardGraphStyle*)dashedLineStartArrow;
// 实线直线，起点实心箭头，终点普通箭 需同时设置setToolType:TEDU_BOARD_TOOL_TYPE_LINE
+ (TEduBoardGraphStyle*)solidLineBothArrow;
// 实心三角形 需同时设置setToolType:TEDU_BOARD_TOOL_TYPE_RIGHT_TRIANGLE
+ (TEduBoardGraphStyle*)solidTriangle;
// 虚线空心矩形 需同时设置setToolType:TEDU_BOARD_TOOL_TYPE_RECT
+ (TEduBoardGraphStyle*)dashedRect;

@end

/**
 * @ingroup def
 * @brief 文件转码参数
*/
@interface TEduBoardTranscodeConfig : NSObject
@property (nonatomic, strong) NSString *minResolution;      ///<指定转码结果最小分辨率，可用于提高转码结果清晰度，如960x540，宽高分隔符为x
@property (nonatomic, assign) bool isStaticPPT;             ///<指定对于 PPT 文件是否启用静态转码（转为静态图片），默认转为 H5 动画（转码耗时较长）
@property (nonatomic, strong) NSString *thumbnailResolution;///<指定文件生成缩略图分辨率，默认不生成缩略图（生成缩略图增加额外转码消耗），如200x200，宽高分隔符为x
@end

/**
 * @ingroup def
 * @brief 文件转码结果
*/
@interface TEduBoardTranscodeFileResult : NSObject
@property (nonatomic, strong) NSString *taskId;                     ///<任务 ID
@property (nonatomic, assign) TEduBoardFileTranscodeStatus status;  ///<转码状态
@property (nonatomic, assign) NSInteger progress;                   ///<转码进度（百分比）
@property (nonatomic, strong) NSString *title;                      ///<文件标题
@property (nonatomic, strong) NSString *resolution;                 ///<文件分辨率，如@"1024x768"
@property (nonatomic, assign) NSInteger pages;                      ///<文件总页数
@property (nonatomic, strong) NSString *url;                        ///<转码结果 URL
@property (nonatomic, strong) NSString *thumbnailResolution;        ///<文件生成缩略图的分辨率，如@"200x200"
@property (nonatomic, strong) NSString *thumbnailUrl;               ///<文件生成缩略图 URL
@end

/**
 * @ingroup def
 * @brief 鼠标样式
*/
@interface TEduBoardAddElementOptions : NSObject
@property (nonatomic, assign) bool erasable;    ///<元素是否支持橡皮擦擦除 默认为true
@property (nonatomic, strong) NSString *left;   ///<元素距左边界的偏移，支持像素或百分比，若不填写单位默认按像素计算
@property (nonatomic, strong) NSString *top;    ///<元素距上边界的偏移，支持像素或百分比，若不填写单位默认按像素计算
@end

/**
 * @ingroup def
 * @brief 鼠标样式
*/
@interface TEduBoardCursorIcon : NSObject
@property (nonatomic, strong) NSString *cursor;///<浏览器内置鼠标指针样式，[可取值参考文档](https://developer.mozilla.org/zh-CN/docs/Web/CSS/cursor)
@property (nonatomic, strong) NSString *url;   ///<自定义鼠标指针样式的 URL，[格式限制参考文档](https://developer.mozilla.org/zh-CN/docs/Web/CSS/cursor/url)
@property (nonatomic, assign) UInt32 offsetX;  ///<自定义鼠标指针样式的横行偏移
@property (nonatomic, assign) UInt32 offsetY;  ///<自定义鼠标指针样式的纵向偏移
@end

/**
 * @ingroup def
 * @brief 快照信息
*/
@interface TEduBoardSnapshotInfo : NSObject
@property (nonatomic, strong) NSString *path;   ///<快照文件存储的本地路径，包括文件名和后缀，只支持 png，例如：aaa/bbb/ccc/xxx.png
@property (nonatomic, assign) BOOL full;   ///<是否截全尺寸白板，默认false。true:全尺寸模式，会忽略白板缩放，截全尺寸白板；false:所见即所得模式，会包含白板缩放，移动，截可见白板的内容
@end


/**
 * @ingroup def
 * @brief 工具提示语样式
*/
@interface TEduBoardToolTypeTitleStyle : NSObject
@property (nonatomic, assign) TEduBoardPosition position;   ///< 显示位置
@property (nonatomic, strong) TEColor *color;               ///< 文本颜色
@property (nonatomic, assign) UInt32 size;                  ///< 实际像素值取值(size * 白板的高度 / 10000)px
@property (nonatomic, assign) TEduBoardTextStyle style;     ///< 文本样式
@end

/**
 * @ingroup def
 * @brief 画板显示区域
*/
typedef struct __attribute__((objc_boxable)) TEBBoundingBox {
    CGFloat x1, y1, x2, y2;  //  画板显示区域的左上角和右下角
} TEBBoundingBox;

/**
 * @ingroup def
 * @brief 数学公式
*/
@interface TEduBoardElementFormula : NSObject
@property (nonatomic, strong) NSString* expression;          ///<公式的函数表达式，注意添加转义字符 示例:@"f(x) = \\\\int_{-\\\\infty}^\\\\infty \\\\hat{f}(\\\\xi)\\\\,e^{2 \\\\pi i \\\\xi x} \\\\,d\\\\xi";
@property (nonatomic, assign) NSInteger fontSize;           ///< 公式字号
@property (nonatomic, strong) TEColor* fontColor;           ///< 公式颜色
@end


/**
 * @ingroup def
 * @brief 数学函数画板
*/
@interface TEduBoardElementMathBoard : NSObject
@property (nonatomic, assign) TEEdgeInsets boundingbox;     ///< 画板显示区域的左上角和右下角, 对应 [x1, y1, x2, y2]; left = x1, top = y1; right = x2, bottom = y2;
@property (nonatomic, strong) TEColor *axisColor;           ///< 画板坐标系颜色
@property (nonatomic, assign) bool axis;                    ///< 是否显示坐标轴
@property (nonatomic, assign) bool grid;                    ///< 是否显示网格
@property (nonatomic, strong) NSString* width;              ///< 画板宽度，支持像素或百分比，若不填写单位默认按像素计算,例如：'300px'
@property (nonatomic, strong) NSString* height;             ///< 画板高度，支持像素或百分比，若不填写单位默认按像素计算,例如：'200px'
@property (nonatomic, assign) bool showControl;             ///< 是否显示控制栏
@property (nonatomic, assign) bool enableZoom;              ///< 是否开启鼠标滚轮缩放（需要按住Shift）
@property (nonatomic, assign) bool enablePan;               ///< 是否开启鼠标拖拽平移（需要按住Shift）
@end

/**
 * @ingroup def
 * @brief 数学函数图像
*/
@interface TEduBoardElementMathGraph : NSObject
@property (nonatomic, strong) NSString* mathBoardId;        ///< 数学函数画板的元素id，
@property (nonatomic, strong) NSString* expression;         ///< 数学函数表达式，例如sin(x)
@property (nonatomic, strong) TEColor *color;               ///< 数学函数图像颜色
@property (nonatomic, strong) TEColor *selectedColor;       ///< 数学函数图像选中时的颜色
@end


/**
 * @ingroup def
 * @brief 水印参数
*/
@interface TEduBoardElementWatermark : NSObject
@property (nonatomic, strong) NSString* content;     ///< 图片水印：https地址；文本水印：普通字符串内容
@property (nonatomic, strong) NSString* left;        ///< 左边，支持像素或百分比，若不填写单位默认按像素计算,例如：'300px'
@property (nonatomic, strong) NSString* top;         ///< 上边，支持像素或百分比，若不填写单位默认按像素计算,例如：'300px'
@property (nonatomic, strong) NSString* width;       ///< 图片水印有效，宽度，支持像素或百分比，若不填写单位默认按像素计算,例如：'300px'
@property (nonatomic, strong) NSString* height;      ///< 图片水印有效，高度，支持像素或百分比，若不填写单位默认按像素计算,例如：'200px'
@property (nonatomic, assign) CGFloat opacity;       ///< 透明度 [0, 1]
@property (nonatomic, assign) NSInteger deg;         ///< 文本水印有效，角度
@property (nonatomic, strong) NSString* color;        ///< 文本水印有效，颜色
@property (nonatomic, strong) NSString* textSize;    ///< 文本水印有效，字体大小
@property (nonatomic, assign) bool fix;              ///< 文本水印有效，true: 静态水印；false: 动态水印
@property (nonatomic, assign) NSInteger interval;    ///< 文本水印有效，当为动态水印时生效，单位毫秒
@end

/**
 * @ingroup def
 * @brief 复制几何图形线条样式
*/
@interface TEduBoardCopyGraphElementLineStyle : NSObject
@property (nonatomic, assign) TEduBoardLineType lineType;       ///< 线类型
@property (nonatomic, assign) TEduBoardFillType fillType;       ///<填充类型, 即实心空心，只对平面几何图形有效，`TEduBoardFillType`
@property (nonatomic, assign) TEduBoardArrowType startArrowType;     ///< 开始箭头类型， 只有直线支持
@property (nonatomic, assign) TEduBoardArrowType endArrowType;       ///< 结束箭头类型， 只有直线支持
@end

/**
 * @ingroup def
 * @brief 复制几何图形样式
*/
@interface TEduBoardCopyGraphElementStyle : NSObject
@property (nonatomic, assign) BOOL enableCustomOffset; ///< 是否自定义复制偏移量，默认false
@property (nonatomic, assign) CGFloat offsetX;         ///自定义X轴复制偏移量，enableCustomOffset为true时生效
@property (nonatomic, assign) CGFloat offsetY;        ///< 自定义Y轴复制偏移量，enableCustomOffset为true时生效
@property (nonatomic, strong) TEColor *color;         ///< 自定义复制图形线条颜色
@property (nonatomic, assign) CGFloat thin;           ///自定义复制图形线条粗细
@property (nonatomic, assign) BOOL solid;             ///< 自定义复制图形线条是否实心
@property (nonatomic, strong) TEduBoardCopyGraphElementLineStyle *style;   ///< 自定义复制图形线条颜色
@end

/**
 * @ingroup def
 * @brief 复制几何图形配置
*/
@interface TEduBoardCopyGraphElementConfig : NSObject
@property (nonatomic, assign) BOOL disableAutoSelect;       ///< 是否禁止复制后自动选中元素，默认false
@end

/**
* @ingroup callback
* @brief 白板事件回调接口
*/
@protocol TEduBoardDelegate <NSObject>
/// @name 通用事件回调
/// @{


/**
 * @brief 白板错误回调
 * @param code              错误码，参见 TEduBoardErrorCode 定义
 * @param msg                错误信息，编码格式为 UTF8
 * @see TEduBoardErrorCode
*/
- (void)onTEBError:(TEduBoardErrorCode)code msg:(NSString *)msg;

/**
 * @brief 白板警告回调
 * @param code              错误码，参见 TEduBoardWarningCode 定义
 * @param msg                错误信息，编码格式为 UTF8
 * @see TEduBoardWarningCode
*/
- (void)onTEBWarning:(TEduBoardWarningCode)code msg:(NSString *)msg;
/// @}

/// @name 基本流程回调
/// @{

/**
 * @brief 白板初始化完成回调
 *
 * 收到该回调后表示白板已处于可正常工作状态（此时白板为空白白板，历史数据尚未拉取到）
 */
- (void)onTEBInit;

/**
 * @brief 白板历史数据同步完成回调
*/
- (void)onTEBHistroyDataSyncCompleted;

@optional
/**
 * @brief 白板同步数据回调
 * @param data                白板同步数据（JSON 格式字符串）
 *
 * 收到该回调时需要将回调数据通过信令通道发送给房间内其他人，接受者收到后调用 addSyncData 接口将数据添加到白板以实现数据同步
 * 该回调用于多个白板间的数据同步，使用腾讯云 IMSDK 进行实时数据同步时，不会收到该回调
*/
- (void)onTEBSyncData:(NSString *)data;

/**
 * @brief 白板可撤销状态改变回调
 * @param canUndo            白板当前是否还能执行 Undo 操作
*/

- (void)onTEBUndoStatusChanged:(bool)canUndo;

/**
 * @brief 白板可重做状态改变回调
 * @param canRedo            白板当前是否还能执行 Redo 操作
*/
- (void)onTEBRedoStatusChanged:(bool)canRedo;
/// @}


/// @name 涂鸦功能回调
/// @{

 /**
  * @brief 白板图片状态改变回调
  * @param boardId       白板 ID
  * @param url                白板图片 URL
  * @param status          新的白板图片状态，注意：如果监听到TEDU_BOARD_IMAGE_STATUS_LOAD_ERROR,
  * TEDU_BOARD_IMAGE_STATUS_LOAD_TIMEOUT事件，请调用refresh 接口进行重新渲染白板。
  */
- (void)onTEBImageStatusChanged:(NSString *)boardId url:(NSString *)url status:(TEduBoardImageStatus)status;

/**
 * @brief 设置白板背景图片回调
 * @param url                调用 SetBackgroundImage 时传入的 URL
 *
 * 只有本地调用 SetBackgroundImage 时会收到该回调
 * 收到该回调表示背景图片已经上传或下载成功，并且显示出来
*/
- (void)onTEBSetBackgroundImage:(NSString *)url;

/**
 * @brief 添加图片元素回调
 * @param url                调用 addImageElement 时传入的 URL
 *
 * 只有本地调用 addImageElement 时会收到该回调
 * 收到该回调表示背景图片已经上传或下载成功，并且显示出来
*/
- (void)onTEBAddImageElement:(NSString *)url DEPRECATED_MSG_ATTRIBUTE("从2.6.2起此接口被废弃，请使用addElement添加元素，并监听onTEBAddElement请求");

/**
 * @brief 添加元素回调
 * @param elementId     调用 addElement 时返回的元素 ID
 * @param url                  调用 addElement 时传入的 url
 * @param type                 元素类型TEduBoardElementType
 *
 * 只有本地调用 addElement 时会收到该回调
 * 收到该回调表示元素添加成功，并且显示出来
*/
- (void)onTEBAddElement:(NSString *)elementId url:(NSString *)url type:(TEduBoardElementType)type;

/**
 * @brief 删除元素回调
 * @param elementIds    删除的元素 IDs
 *
 * 只有本地调用 addElement 时会收到该回调
*/
- (void)onTEBRemoveElement:(NSArray *)elementIds;

/**
 * @brief 设置白板背景 H5 状态改变回调
 * @param boardId      白板 ID
 * @param url               白板图片 URL
 * @param status         新的白板图片状态
*/
- (void)onTEBBackgroundH5StatusChanged:(NSString *)boardId url:(NSString *)url status:(TEduBoardBackgroundH5Status)status;
/// @}


/// @name 白板操作回调
/// @{

/**
 * @brief 增加白板页回调
 * @param boardIds        增加的白板页 ID 列表（使用后不需要自行调用 Release 方法释放，SDK 内部自动释放）
 * @param fileId            增加的白板页所属的文件 ID（目前版本只可能为#DEFAULT）
 */
- (void)onTEBAddBoard:(NSArray *)boardIds fileId:(NSString *)fileId;

/**
 * @brief 删除白板页回调
 * @param boardIds            删除的白板页 ID（使用后不需要自行调用 Release 方法释放，SDK 内部自动释放）
 * @param fileId                删除的白板页所属的文件 ID（目前版本只可能为#DEFAULT）
*/
- (void)onTEBDeleteBoard:(NSArray *)boardIds fileId:(NSString *)fileId;

/**
 * @brief 跳转白板页回调
 * @param boardId            跳转到的白板页 ID
 * @param fileId            跳转到的白板页所属的文件 ID
 * @param userId            操作者 ID 
*/
- (void)onTEBGotoBoard:(NSString *)boardId fileId:(NSString *)fileId userId:(NSString *)userId;

/**
 * @brief 白板页动画步数回调
 * @param currentStep        当前白板页动画步数，取值范围 [0, totalStep)
 * @param totalStep            当前白板页动画总步数
*/
- (void)onTEBGotoStep:(uint32_t)currentStep totalStep:(uint32_t)totalStep;

/**
 * @brief 框选工具选中回调
 *
 * 只有框选中涂鸦或图片元素后触发回调
*/
- (void)onTEBRectSelected  DEPRECATED_MSG_ATTRIBUTE("从2.6.2起此回调被废弃，请使用 选中元素回调 onTEBSelectedElements");

/**
 * @brief 刷新白板回调
*/
- (void)onTEBRefresh;

/**
 * @brief 白板快照
 * @param path 快照本地路径
 * @param code 错误码，返回 0，表示获取快照成功
 * @param msg 错误信息
*/
- (void)onTEBSnapshot:(NSString *)path errorCode:(TEduBoardErrorCode)code errorMsg:(NSString *)msg;
/// @}


/// @name 文件操作回调
/// @{

/**
 * @brief 增加转码文件回调
 * @param fileId            增加的文件 ID
 *
 * 文件加载完成后才会触发该回调
*/
- (void)onTEBAddTranscodeFile:(NSString *)fileId;

/**
 * @brief 删除文件回调
 * @param fileId            删除的文件 ID
*/
- (void)onTEBDeleteFile:(NSString *)fileId;

/**
 * @brief 切换文件回调
 * @param fileId            切换到的文件 ID
*/
- (void)onTEBSwitchFile:(NSString *)fileId;

/**
 * @brief 文件上传进度回调
 * @param path                      正在上传的文件路径
 * @param currentBytes     当前已上传大小，单位 bytes
 * @param totalBytes          文件总大小，单位 bytes
 * @param uploadSpeed       文件上传速度，单位 bytes
 * @param percent            文件上传进度，取值范围 [0, 1]
*/
- (void)onTEBFileUploadProgress:(NSString *)path
                currentBytes:(int)currentBytes
                  totalBytes:(int)totalBytes
                 uploadSpeed:(int)uploadSpeed
                    percent:(float)percent;

/**
 * @brief 文件上传状态回调
 * @param path                      正在上传的文件路径
 * @param status                 文件上传状态
 * @param errorCode         文件上传错误码
 * @param errorMsg            文件上传错误信息
*/
- (void)onTEBFileUploadStatus:(NSString *)path
                       status:(TEduBoardUploadStatus)status
                    errorCode:(int)errorCode
                    errorMsg:(NSString *)errorMsg;

/**
 * @brief H5 文件状态回调
 * @param fileId            文件 ID
 * @param status            文件状态
*/
- (void)onTEBH5FileStatusChanged:(NSString *)fileId
                       status:(TEduBoardH5FileStatus)status;

/**
 * @brief 视频文件状态回调
 * @param fileId            文件 ID
 * @param status            文件状态
 * @param progress       当前进度（秒）（仅支持 mp4 格式）
 * @param duration            总时长（秒）（仅支持 mp4 格式）
*/
- (void)onTEBVideoStatusChanged:(NSString *)fileId
                         status:(TEduBoardVideoStatus)status
                        progress:(CGFloat)progress
                        duration:(CGFloat)duration;

/**
 * @brief 音频文件状态回调
 * @param elementId      元素 ID
 * @param status             文件状态
 * @param progress        当前进度（秒）
 * @param duration            总时长（秒）
*/
- (void)onTEBAudioStatusChanged:(NSString *)elementId
                         status:(TEduBoardAudioStatus)status
                        progress:(CGFloat)progress
                        duration:(CGFloat)duration;

/**
 * @brief 增加批量图片文件回调
 * @param fileId            增加的文件 ID
 *
 * 文件加载完成后会触发该回调
 */
- (void)onTEBAddImagesFile:(NSString *)fileId;

/**
 * @brief 元素锁定变更事件
 * @param elementIds   ID 列表
 * @param locked           状态
*/
- (void)onTEBBoardElementLockedChanged:(NSArray<NSString *> *)elementIds
                           locked:(BOOL)locked;

/**
 * @brief H5PPT 文件状态改变回调
 * @param fileId           文件 ID
 * @param status           文件状态
 * @param message         状态消息
*/
- (void)onTEBH5PPTStatusChanged:(NSString *)fileId
                         status:(TEduBoardH5PPTStatus)status
                        message:(NSString *)message;

/**
 * @brief H5PPT 媒体状态改变回调
 * @param fileId           文件 ID
 * @param mediaId          播放源 ID
 * @param status           媒体状态
 * @param currentTime      当前播放进度（秒）
*/
- (void)onTEBH5PPTMediaStatusChanged:(NSString *)fileId
                             mediaId:(NSString *)mediaId
                              status:(TEduBoardH5PPTMediaStatus)status
                         currentTime:(double)currentTime;


/**
 * @brief 文本组件状态回调
 * @param elementId           文本组件id
 * @param status              文本组件状态（focus：获得焦点，blur：失去焦点）
 * @param value               文本内容
 * @param left                文本组件水平偏移
 * @param top                 文本组件垂直偏移
*/
- (void)onTEBTextElementStatusChanged:(NSString *)elementId
                               status:(NSString *)status
                                value:(NSString *)value
                                 left:(float)left
                                  top:(float)top;

/**
 * @brief 白板图片元素加载状态
 * @param boardId               白板 ID
 * @param url                   白板图片 URL
 * @param elementId             当前元素id
 */
- (void)onTEBImageElementStatusChanged:(NSString *)boardId url:(NSString *)url elementId:(NSString*)elementId;

/**
 * @brief 白板图片元素加载状态
 * @param code               白板文字工具异常状态码
 * @param msg                异常信息
 */
- (void)onTEBTextElemenWarning:(TEduBoardTextComponentStatus)code msg:(NSString *)msg;

/**
 * @brief 框选工具选中元素回调
 * @param selectdElements          被选中元素数组
 */
- (void)onTEBSelectedElements:(NSArray<TEduBoardElementInfo *> *) selectdElements;

/**
 * @brief 元素位置发生改变回调
 * @param elements          位置发生变化的元素数组
 * @param status               位置变化状态
 */
- (void)onTEBElementsPositonChanged:(NSArray<TEduBoardElementInfo *> *) elements status:(TEduBoardPositionChangeStatus)status;
/**
 * @brief 数学函数图像工具事件
 * @param code                       元素ID
 * @param boardId                    函数画板ID
 * @param graphId                    函数图像ID
 * @param message                    异常信息
 */
- (void)onTEBMathGraphEvent:(TEduBoardMathGraphCode)code
                    boardId:(NSString *)boardId
                    graphId:(NSString *)graphId
                    message:(NSString *)message;

/**
 * @brief 远端白板缩放移动状态回调
 * @param fid                       文件ID
 * @param scale                     文件缩放比
 * @param xOffset                   当前可视区域距左上角的横向偏移量
 * @param yOffset                   当前可视区域距左上角的纵向偏移量
 */
- (void)onTEBZoomDragStatus:(NSString *)fid
                      scale:(int)scale
                    xOffset:(int)xOffset
                    yOffset:(int)yOffset;


/**
 * @brief 白板离线告警
 * @param count                       告警次数
 */
- (void)onTEBOfflineWarning:(NSInteger)count;

/**
 * @brief 分组讨论状态变更
 * @param enable                       分组模式状态
 * @param groupId                      发生变化的分组id
 * @param operationType                `TEduBoardClassGroupOperationType`, 触发状态变更的操作
 * @param message                      操作信息
 */
- (void)onTEBGroupStatusChanged:(bool)enable
                        groupId:(NSString *)groupId
                  operationType:(TEduBoardClassGroupOperationType)operationType
                        message:(NSString *)message;

/**
 * @brief 白板移动回调
 * @param boardId                      分组模式状态
 * @param trigger                      事件触发来源
 * @param scrollLeft                   左侧滚动百分比距离
 * @param scrollTop                    顶部滚动百分比距离
 * @param scale                        白板缩放比
 */
- (void)onTEBBoardScrollChanged:(NSString *)boardId
                        trigger:(NSInteger)trigger
                     scrollLeft:(double)scrollLeft
                      scrollTop:(double)scrollTop
                          scale:(double)scale;

/**
 * @brief 白板缩放回调
 * @param boardId                      分组模式状态
 * @param scale                        白板缩放比
 * @param xOffset                      当前可视区域距左上角的横向偏移量
 * @param yOffset                      当前可视区域距左上角的纵向偏移量
 */
- (void)onTEBBoardScaleChanged:(NSString *)boardId
                         scale:(double)scale
                       xOffset:(double)xOffset
                       yOffset:(double)yOffset;

/**
 * @brief                              自定义鼠标样式时,坐标位置回调
 * @param position                     鼠标坐标位置
 */
- (void)onTEBBoardCursorPosition:(CGPoint)position;

/**
 * @brief                              权限发生变更事件
 * @param permissions                  权限名列表
 * @param filters                      约束条件，当filters为无内容时 则表示对应的permissions将不再鉴权
 * 如调用 teduBoard.enablePermissionChecker(["Element::Update::MathTool", "Element::Update::Text"], ['operator/A,B', 'creator/C,D'])，则会触发该回调，回调内容为
 * permissions = ["Element::Update::MathTool", "Element::Update::Text"],
 * filters = {
 *    "operator": ['A', 'B']
 *    "creator": ['C', 'D']
 * }
 * 如调用 teduBoard.disablePermissionChecker(["Element::Update::MathTool"])，则会触发该回调，回调内容为
 * permissions = ["Element::Update::MathTool"],
 * filters = {}

 */
- (void)onTEBBoardPermissionChanged:(NSArray<NSString *> *)permissions filters:(NSDictionary *)filters;

/**
 * @brief                           执行操作时权限被拒
 * @param permission                被拒的权限
 */
- (void)onTEBBoardPermissionDenied:(NSString *)permission;

/**
 * @brief    导入数据完成回调
 * @param status               状态码
 */
- (void)onTEBImportInLocalModeCompleted:(TEduBoardImportInLocalModeStatus)status;

/**
 * @brief     绘画动作状态切换事件
 * @param status   `TEduBoardDrawStatusChanged` 绘画动作状态码
 */
- (void)onTEBBoardDrawStatusChanged:(TEduBoardDrawStatusChanged)status;

/**
 * @brief     h5ppt页面降级渲染事件通知
 * @param filedId   降级渲染的ppt文件id
 *@param slideIndex   降级渲染的ppt文件id
 */
- (void)onTEBBoardH5pptDownGrade:(NSString *)filedId slideIndex:(NSInteger)slideIndex;
@end
