//
//  TEduBoardController.h
//  TEduBoard
//
//  Created by kennethmiao on 2019/4/8.
//  Copyright © 2019年 kennethmiao. All rights reserved.
//

#import <Foundation/Foundation.h>
#import "TEduBoardDef.h"

/**
 * @defgroup ctrl TEduBoardController
 * 白板功能的主要接口类
 */

/**
 * @defgroup callback TEduBoardDelegate
 * 白板功能的回调接口类
 */

/**
 * @defgroup def 关键类型定义
 */

/**
 * @defgroup code 错误码
 */

/**
 * @ingroup ctrl
 * @brief 白板控制器
*/
@interface TEduBoardController : NSObject
- (instancetype)init __attribute__((unavailable("Please use initWithParam:roomId:timSync initialize the TEduBoardView")));
- (instancetype)new __attribute__((unavailable("Please use initWithParam:roomId:timSync initialize the TEduBoardView")));
- (instancetype)initWithFrame:(CGRect)frame __attribute__((unavailable("Please use initWithParam:roomId:timSync initialize the TEduBoardView")));

/// @name 设置 TEduBoardCallback 回调
/// @{

/**
 * @brief 设置事件回调监听
 * @param delegate           事件回调监听, `TEduBoardDelegate`
 * @warning 建议在调用 `[TEduBoardController initWithAuthParam:roomId:initParam:]` 之后马上调用该方法以支持错误处理
 */
- (void)addDelegate:(id<TEduBoardDelegate>)delegate;

/**
 * @brief 删除事件回调监听
 * @param delegate             事件回调监听 `TEduBoardDelegate`
*/
- (void)removeDelegate:(id<TEduBoardDelegate>)delegate;

/// @}


/// @name 基本流程接口
/// @{

/**
 * @brief 初始化白板
 * @param authParam         授权参数 `TEduBoardAuthParam`
 * @param initParam         可选参数，`TEduBoardInitParam` 指定用于初始化白板的一系列属性值
 * @param roomId            课堂 ID
 * @warning 使用腾讯云 IMSDK 进行实时数据同步时，只支持一个白板实例，创建多个白板实例可能导致涂鸦状态异常
 *
 * 可用 initParam.timSync 指定是否使用腾讯云IMSDK进行实时数据同步
 * initParam.timSync == true 时，会尝试反射调用腾讯云 IMSDK 作为信令通道进行实时数据收发（只实现消息收发，初始化、进房等操作需要用户自行实现），目前仅支持 IMSDK 4.3.118 及以上版本
 */
- (instancetype)initWithAuthParam:(TEduBoardAuthParam *)authParam roomId:(NSInteger)roomId initParam:(TEduBoardInitParam *)initParam;

/**
* @brief 设置用户信息
* @param userInfo         用户信息 `TEduBoardUserInfo`
*/
- (void)setUserInfo:(TEduBoardUserInfo *)userInfo;

/**
* @brief 反初始化白板
* @warning 调用反初始化接口后会释放内部资源，白板功能将失效。
*/
- (void)unInit;

/**
 * @brief 获取白板渲染 View
 * @return 白板渲染 View
*/
- (TEView *)getBoardRenderView;

/**
 * @brief 添加白板同步数据
 * @param data                 接收到的房间内其他人发送的同步数据
 *
 * 该接口用于多个白板间的数据同步，使用内置 IM 作为信令通道时，不需要调用该接口
*/
- (void)addSyncData:(NSString *)data;

/**
 * @brief 设置白板是否开启数据同步
 * @param enable    是否开启
 *
 * 白板创建后默认开启数据同步，关闭数据同步，本地的所有白板操作不会同步到远端和服务器
*/
- (void)setDataSyncEnable:(BOOL)enable;

/**
 * @brief 获取白板是否开启数据同步
 * @return 是否开启数据同步，true 表示开启，false 表示关闭
*/
- (BOOL)isDataSyncEnable;


/**
 * @brief 开启多点触控
 * @param enable    是否开启
 *
 *
*/
- (void)enableMultiTouch:(BOOL)enable;

/**
 * @brief 获取同步时间戳
 * @return 毫秒级同步时间戳
*/
- (uint64_t)getSyncTime;

/**
 * @brief 同步远端时间戳
 * @param userId                远端用户 ID
 * @param timestamp         远端用户毫秒级同步时间戳
*/
- (void)syncRemoteTime:(NSString *)userId timestamp:(uint64_t)timestamp;

/**
 * @brief 添加资源主备域名映射
 * 1.主备域名均需包含协议类型(支持http/https)
 * 2.切换域名重试超时默认为5s
 * 3.多次调用此接口，可为统一主域名添加多个备用域名。如果备用域名已在列表中则不再添加
 * 4.当主域名不可用时，SDK将按从前往后的顺序从列表中选择一个备用域名并重试，请务必将可用性高的域名添加到列表前面
 *
 * @param domain 主域名 (必填）
 * @param backup 备用域名 (必填)
 * @param priority 优先级，备用域名优先级相同时，按添加的顺序决定优先级，添加越前优先级越高
 */
- (void)addBackupDomain:(NSString *)domain backup:(NSString *)backup priority:(NSInteger)priority;

/**
 * @brief 删除资源主备域名映射
 * @param domain 主域名 (必填）
 * @param backup 备用域名 (必填)
*/
- (void)removeBackupDomain:(NSString *)domain backup:(NSString *)backup;

/**
 * @brief 重置白板
 *
 * 调用该接口后将会删除所有的白板页和文件
*/
- (void)reset;

/**
 * @brief 获取版本号
 * @return NSString  版本号字符串
*/
+ (NSString *)getVersion;

/**
 * @brief 启用和关闭缓存, 请在`TEduBoardController`初始化之前调用
 * @param enable 是否启用，默认为 YES
 */
+ (void)setEnableCache:(bool)enable;

/**
 * @brief 获取缓存的启用状态
 * @return 是否启用
 */
+ (BOOL)enableCache;

/**
 * @brief 是否允许白板sdk收集网络信息，用于问题诊断和网络优化
 */
+ (void)setCollectNetInfo:(bool)enable;


/**
 * @brief 获取白板sdk是否收集网络信息
 * @return 是否启用
 */
+ (BOOL)enableCollectNetInfo;

/**
 * @brief 是否开启单实例模式
 * @param singleMode 单实例模式，true为单例(第二个TEduBoardController实例初始化时会回收旧的实例)，false则容许多实例
 * @warning SDK默认为单实例模式
 */
+ (void)setSingleMode:(bool)singleMode;

/**
 * @brief 获取单实例模式
 * @return 是否启用单实例模式
 */
+ (BOOL)isSingleMode;

/**
 * @brief 设置输出日志级别
 *
 * @param level 输出日志的级别 `TEduBoardLogLevel`
 * @warning 建议在白板实例创建后立即调用
 * @warning 低于 level 的日志将不会输出
*/
- (void)setLogLevel:(TEduBoardLogLevel)level;

/**
 * 修改信令同步频率，该值的允许范围为 [5, 20]
 * 开启了多指操控后，如果频率不提高，消息会有延时会比较大
 * @param  fps 频率
 */
- (void)setSyncFps:(NSUInteger)fps;
/// @}

/**
 *
 *权限命令
类型::操作::属性/对象
 
// 设置背景颜色
Background::Update::Color
// 设置背景webview
Background::Update::Frame
// 设置背景图片
Background::Update::Image
// 添加白板
Board::Add
// 清空白板
Board::Clear
// 删除白板
Board::Delete
// 缩放白板
Board::Scale::*
// 切换白板
Board::Switch::Page
// 切换步数
Board::Switch::Step
// 更新白板内容填充模式
Board::Update::ContentFitMode
// 更新白板比例
Board::Update::Ratio
// 添加元素
Element::Add::CustomGraph
Element::Add::Graph
Element::Add::Highlighter
Element::Add::Laser
Element::Add::Pen
Element::Add::Text
Element::Add::MathTool
Element::Add::*
// 删除元素
Element::Delete::*
// 选择元素
Element::Select::*
// 更新元素
Element::Update
// 添加文件
File::Add
// 清空文件
File::Clear
// 删除文件
File::Delete
// 切换文件
File::Switch
// 更新文件（缩放，音视频进度）
File::Update::Audio
File::Update::Scale
File::Update::Video
// 设置实例属性
Instance::Configure::ClassGroup
Instance::Configure::Snapshot
Instance::Configure
 *
 */

/**
 *筛选器*
 *
 属性名/满足条件的值
 creator/user1  仅允许具有creator属性，且值为user1的对象进行操作
 operator/user2  仅允许操作者为user2进行操作
 */

/**
 *设置示例
 *
 */

//### 场景1
//房间里有教师T、学生A。赋予 T 文件，白板，元素操作权限，禁用 A 的全部权限
/** T/A 设置 */
//enablePermissionChecker(['File::*::*', 'Board::*::*', 'Element::*::*'], ['operator/T']);

//### 场景2
//房间里有教师T、学生A、学生B。TAB 均赋予元素的操作权限
/** 方式一 */
// 三人均设置
//enablePermissionChecker(['Element::*::*'], ['operator/A,B,T']);
/** 方式二 */
// T
//enablePermissionChecker(['Element::*::*'], ['operator/T']);
// A
//enablePermissionChecker(['Element::*::*'], ['operator/A']);
// B
//enablePermissionChecker(['Element::*::*'], ['operator/B']);

//### 场景3
//房间里有教师T、学生A、学生B。T能删除所有人创建的元素（T自己、A、B），A只能删除A自己创建的元素, 不能删除其他人的创建的元素；
// T
//enablePermissionChecker(['Element::Delete::*'], ['creator/T,A']);
// AB不需要设置

//### 场景4
//房间里有教师T、学生A、学生B。T 只能操作 B 创建的元素,
// T
//enablePermissionChecker(['Element::*::*'], ['creator/B']);

//### 场景5
//禁用当前用户的全部操作
//enablePermissionChecker(['*::*::*'], ['operator/']);
// 'operator/' 表示对于操作者（operator）这个条件，符合的值是空的，意味着没有操作者能够被允许通过这个权限校验

//### 场景6
//启用当前用户的全部操作
//enablePermissionChecker(['*::*::*'], ['operator/*']);
// 'operator/*' 表示对于操作者（operator）这个条件，符合的值是任意的，意味着所有操作者能够被允许通过这个权限校验

/**
 * 启用互动白板操作权限校验，注意仅对当前用户生效
 * @param  permissions 权限列表
 * @param  filters 过滤规则列表，
 *
 */
- (void)enablePermissionChecker:(NSArray<NSString *> *)permissions filters:(NSArray<NSString *> *)filters;

/**
 * @brief 关闭互动白板操作权限校验
 * @param  permissions 权限列表
 */
- (void)disablePermissionChecker:(NSArray<NSString *> *)permissions;
/**
 * @brief 重置互动白板操作权限校验
 */
- (void)resetPermissionChecker;

/**
 * @brief 禁止白板响应鼠标事件
 * @param  disable true 表示禁止白板响应鼠标事件  false 表示允许响应鼠标事件
 */
- (void)disablePointerEventResponding:(BOOL)disable;


/// @name 涂鸦相关接口
/// @{

/**
 * @brief 设置允许操作哪些用户绘制的图形
 * 1.ERASER 工具只能擦除 users 参数列出的用户绘制的涂鸦，无法擦除其他人绘制的涂鸦
 * 2.POINTSELECT、SELECT 工具只能选中 users 参数列出的用户绘制的涂鸦，无法选中其他人绘制的涂鸦
 * 3.clear 接口只能用于清空选中涂鸦以及 users 参数列出的用户绘制的涂鸦，无法清空背景及其他人绘制的涂鸦
 * 4.白板包含的其他功能未在本列表明确列出者都可以确定不受本接口影响
 *    
 * @param users             【必填】指定允许操作的用户集，为nil或@[]表示不加限制, @[@'']表示限制所有用户
 * @param operationType     【选填】指定允许的操作类型 `TEduBoardAccessibleOperation`
 *
*/
- (void)setAccessibleUsers:(NSArray<NSString *> *)users operationType:(NSArray<NSNumber *> *)operationType;

/**
 * @brief 设置白板是否允许涂鸦
 * @param enable             是否允许涂鸦，true 表示白板可以涂鸦，false 表示白板不能涂鸦
 * 白板创建后默认为允许涂鸦状态
 * 2.8.0开始对应权限如下
 * 'Element::Add::*',
 * 'Element::Delete::*',
 * 'Element::Move::*',
 * 'Element::Select::*',
 * 'Element::Update::*',
 * 'Element::Scale::*',
 * 'Element::Rotate::*',
 * 'Background::Update::*',
 * 'Board::Switch::*',
 * 'Board::Clear::*',
 * 'File::Clear::*',
*/
- (void)setDrawEnable:(BOOL)enable;

/**
 * @brief 是否显示涂鸦
 * @param enable             是否显示涂鸦，true 表示白板可以显示涂鸦，false 表示白板不能显示涂鸦
 *
 * 白板创建后默认为允许涂鸦状态
*/
- (void)enableShowGraffiti:(BOOL)enable;

/**
 * @brief 是否支持视频文件涂鸦
 * @param enable             是否能涂鸦，true 表示可以涂鸦，false 表示不能涂鸦
 *
 * 白板创建后默认为不允许视频文件涂鸦状态
*/
- (void)setVideoFileDrawEnable:(BOOL)enable;

/**
 * @brief 获取白板是否允许涂鸦
 * @return 是否允许涂鸦，true 表示白板可以涂鸦，false 表示白板不能涂鸦
*/
- (BOOL)isDrawEnable;

/**
 * @brief 设置所有白板的背景色
 * @param color             要设置的全局背景色
 *
 * 调用该接口将导致所有白板的背景色发生改变
 * 新创建白板的默认背景色取全局背景色
*/
- (void)setGlobalBackgroundColor:(TEColor *)color;

/**
 * @brief 获取白板全局背景色
 * @return 全局背景色
*/
- (TEColor *)getGlobalBackgroundColor;

/**
 * @brief 设置当前白板页的背景色
 * @param color             要设置的背景色
 *
 * 白板页创建以后的默认背景色由 SetDefaultBackgroundColor 接口设定
*/
- (void)setBackgroundColor:(TEColor *)color;

/**
 * @brief 设置白板父容器的背景色
 * @param color             要设置的背景色
 *
*/
- (void)setBoardContainerColor:(TEColor *)color;

/**
 * @brief 获取当前白板页的背景色
 * @return 当前白板页的背景色
*/
- (TEColor *)getBackgroundColor;

/**
 * @brief 设置当前白板页的背景图片
 * @param url                 要设置的背景图片 URL，编码格式为 UTF8
 * @param mode                要使用的图片填充对齐模式, `TEduBoardImageFitMode`
 *
 * 当URL是一个有效的本地文件地址时，该文件会被自动上传到 COS
*/
- (void)setBackgroundImage:(NSString *)url mode:(TEduBoardImageFitMode)mode;

/**
 * @brief 设置背景图片旋转角度
 * @param angle      旋转角度（0-360）
 *
*/
- (void)setBackgroundImageAngle:(NSInteger)angle;

/**
 * @brief 获取背景图片信息
 * @return  背景图片信息
 *
*/
- (TEduBoardBackgroundImageInfo *)getBackgroundImage;

/**
 * @brief 设置当前白板页的背景 H5 页面
 * @param url                要设置的背景 H5 页面 URL
 *
 * 该接口与 SetBackgroundImage 接口互斥
*/
- (void)setBackgroundH5:(NSString *)url;

/**
 * @brief 设置要使用的白板工具
 * @param type                 要设置的白板工具, `TEduBoardToolType`
*/
- (void)setToolType:(TEduBoardToolType)type;

/**
 * @brief 获取正在使用的白板工具
 * @return 正在使用的白板工具
*/
- (TEduBoardToolType)getToolType;

/**
 * @brief 设置图形样式
 * @param style             【必填】要设置的图形样式 `TEduBoardGraphStyle`
 * 示例请参阅demo的 绘制无箭头的虚线直线、绘制终点实心箭头的实线直线、设置虚线直线，起点普通箭头、设置实线直线，起点实心箭头，终点普通箭头、绘制实心三角形、绘制虚线空心矩形
*/
- (void)setGraphStyle:(TEduBoardGraphStyle *)style;

/**
 * @brief 获取图形样式
 * @return 图形样式 `TEduBoardGraphStyle`
*/
- (TEduBoardGraphStyle *)getGraphStyle;
/**
 * @brief 是否启用点擦（分段擦除）功能     2.6.8新增
 * @param enable        是否开启点擦
 * @warning 1. 只有2.6.8+的版本画线才能支持点擦，低于2.6.8以下版本的画线会整条线被擦除
 * @warning 2. 观看端如果是低于2.6.8的版本，实时是看不到点擦变化，重新进课堂后能看到
 *
*/
- (void)setPiecewiseErasureEnable:(BOOL)enable;

/**
 * @brief 获取分段擦除模式开启状态   2.6.8新增
 *
*/
- (BOOL)isPiecewiseErasureEnable;

/**
 * @brief 设置橡皮擦大小     2.6.8新增
 * @param size        橡皮擦大小，单位：像素
 *
*/
- (void)setEraserSize:(NSInteger)size;

/**
 * @brief 获取橡皮擦大小
 * @return 橡皮擦大小
*/
- (NSInteger)getEraserSize;

/**
 * @brief 设置画笔颜色
 * @param color             要设置的画笔颜色
 *
 * 画笔颜色用于所有涂鸦绘制
*/
- (void)setBrushColor:(TEColor *)color;

/**
 * @brief 获取画笔颜色
 * @return 画笔颜色
*/
- (TEColor *)getBrushColor;

/**
 * @brief 设置画笔粗细
 * @param thin                 要设置的画笔粗细
 *
 * 画笔粗细用于所有涂鸦绘制，实际像素值取值(thin * 白板的高度 / 10000)px，如果结果小于1px，则涂鸦的线条会比较虚
*/
- (void)setBrushThin:(UInt32)thin;

/**
 * @brief 获取画笔粗细
 * @return 画笔粗细
*/
- (UInt32)getBrushThin;

/**
 * @brief 设置文本颜色
 * @param color             要设置的文本颜色
*/
- (void)setTextColor:(TEColor *)color;

/**
 * @brief 获取文本颜色
 * @return 文本颜色
*/
- (TEColor *)getTextColor;

/**
 * @brief 设置荧光笔颜色，画笔颜色用于所有荧光笔绘制
 * @param color             要设置的荧光笔颜色
*/
- (void)setHighlighterColor:(TEColor *)color;

/**
 * @brief 获取荧光笔颜色
 * @return 荧光笔颜色
*/
- (TEColor *)getHighlighterColor;
/**
 * @brief 设置文本样式
 * @param style             要设置的文本样式 `TEduBoardTextStyle`
*/
- (void)setTextStyle:(TEduBoardTextStyle)style;

/**
 * @brief 获取文本样式
 * @return 文本样式
*/
- (TEduBoardTextStyle)getTextStyle;

/**
 * @brief 设置文本大小
 * @param size                 要设置的文本大小
 *
 * 实际像素值取值(size * 白板的高度 / 10000)px
*/
- (void)setTextSize:(UInt32)size;

/**
 * @brief 获取文本大小mBoard
 * @return 文本大小
*/
- (UInt32)getTextSize;

/**
 * @brief 设置文本框行高
 * @param lineHeight 文本的行高, 可以设置为1, 1.15, 1.3, 1.5, 2, 3六个档位
 *
*/
- (void)setTextLineHeight:(float)lineHeight;

/**
 * @brief 获取文本框行高
 * @return 文本框行高
*/
- (float)getTextLineHeight;


/**
 * @brief 设置直线样式
 * @param style             要设置的直线样式 `TEduBoardLineStyle`
*/
- (void)setLineStyle:(TEduBoardLineStyle *)style;

/**
 * @brief 获取直线样式
 * @return 直线样式
*/
- (TEduBoardLineStyle *)getLineStyle;

/**
 * @brief 设置椭圆绘制模式
 * @param mode             要设置的椭圆绘制模式 `TEduBoardOvalDrawMode`
*/
- (void)setOvalDrawMode:(TEduBoardOvalDrawMode)mode;

/**
 * @brief 获取椭圆绘制模式
 * @return 椭圆绘制模式
*/
- (TEduBoardOvalDrawMode)getOvalDrawMode;

/**
 * @brief 是否启用原生系统光标
 * @param enable 启用或禁用，默认禁用
*/
- (void)setSystemCursorEnable:(BOOL)enable;


/**
 * @brief 撤销当前白板页上一次动作
*/
- (void)undo;

/**
 * @brief 重做当前白板页上一次撤销
*/
- (void)redo;

/**
 * @brief 清除涂鸦，同时清空背景色以及背景图片
 */

- (void)clear;

/**
 * @brief 清除涂鸦
 */
- (void)clearDraws;

/**
 * @brief 清空当前白板页涂鸦
 * @param background     是否同时清空背景色以及背景图片
 * @param selected 是否只清除选中部分涂鸦
 * @warning 目前不支持清除选中部分的同时清除背景
*/
- (void)clearBackground:(BOOL)background andSelected:(BOOL)selected;

/**
 * @brief 自定义白板工具鼠标样式
 * @param toolType              要设置鼠标样式的白板工具类型
 * @param cursorIcon          要设置的鼠标样式 `TEduBoardCursorIcon`
*/
- (void)setCursorIcon:(TEduBoardToolType)toolType cursorIcon:(TEduBoardCursorIcon *)cursorIcon;

/**
   * 设置光标位置
   * @param positon 光标的偏移位置
   */
- (void)setCursorPosition:(CGPoint)positon;

/**
 * @brief 设置画笔和激光笔工具提示语
 * @param title         提示语
 * @param style          如果为空，则使用默认样式 `TEduBoardToolTypeTitleStyle`
 * @param toolType       工具类型 `TEduBoardToolType`
*/
- (void)setToolTypeTitle:(NSString *)title style:(TEduBoardToolTypeTitleStyle *)style toolType:(TEduBoardToolType)toolType;


/**
 * @brief 设置几何画板元素
 * @param graphType       几何元素类型 `TEduBoardMathGraphType`
 * @param mouseTool         是否切换到鼠标工具
 * @see TEduBoardMathGraphType
*/
- (void)setMathGraphType:(TEduBoardMathGraphType)graphType mouseTool:(BOOL)mouseTool;

/**
 * @brief 使用数学教具
 * @param mathToolType       【必填】要使用的数学教具`TEduMathToolType`
*/
- (void)useMathTool:(TEduMathToolType)mathToolType;

/**
 * @brief 设置鼠标工具行为
 * @param config        配置 `TEduBoardMouseToolBehavior`
*/
- (void)setMouseToolBehavior:(TEduBoardMouseToolBehavior *)config;


/**
 * @brief 是否本地显示自己的别名
 * @param visiable        设置为false时，画笔部分不显示自己的NickName;
*/
- (void)setOwnerNickNameVisible:(BOOL)visiable;

/**
 * @brief 设置自定义字体
 * @param fontFace        字体样式名称
 * @warning 白板内置字体的样式为 'customFontFace'
*/
- (void)setTextFontFamily:(NSString *)fontFace;

/**
 * @brief 添加自定义字体
 * @param fontFace        字体样式名称
 * @param fontUrl        字体资源链接
 * @warning 建议在创建白板实例后立刻调用
 * @warning 本地端和远端都需要设置字体样式，否则可能出现两端字体显示样式不同步
 * @warning 添加系统内置字体时，fontUrl可以传入空字符串
*/
- (void)addTextFontFamily:(NSString *)fontFace fontUrl:(NSString *)fontUrl;

/**
 * @brief 获取当前使用字体
 * @return 当前使用字体
*/
- (NSString *)getTextFontFamily;

/// @}

/// @name 白板页操作接口
/// @{

/**
 * @brief 增加一页白板
 * @param url                要使用的背景图片 URL，编码格式为 UTF8，为 nullptr 表示不指定背景图片
 * @return 白板ID
 * @warning 白板页会被添加到默认文件（文件 ID 为#DEFAULT)，自行上传的文件无法添加白板页
 *
 * 返回值内存由SDK内部管理，用户不需要自己释放
 */
- (NSString *)addBoardWithBackgroundImage:(NSString *)url  DEPRECATED_MSG_ATTRIBUTE("接口将被废弃，请使用addBoard");

/**
 * @brief 设置白板背景图       2.6.8新增
 * @param info         图片信息
 *
 */
-(void)setGlobalBackgroundPic:(TEduBoardBackgroundImageInfo *)info;

/**
 * @brief 获取白板背景图       2.6.8新增
 * @return object
 *
 */
- (TEduBoardBackgroundImageInfo *)getGlobalBackgroundPic;
/**
 * @brief 增加一页白板
 * @param url                【可选】要使用的背景图片url，为 null 表示不指定背景图片，只支持https协议的图片url
 * @param model              【可选】要使用的图片填充对齐模式`TEduBoardImageFitMode`，当设置url时有效
 * @param type               【可选】背景类型，`TEduBoardBackgroundType`
 * @param needSwitch         【可选】是否切换到添加的白板页
 * @warning 1、白板页会被添加到默认文件（文件 ID 为#DEFAULT)，自行上传的文件无法添加白板页;
 * @warning 2、触发 TEduBoard.EVEN.TEB_ADDBOARD 事件
 *
 * 返回值内存由SDK内部管理，用户不需要自己释放
 */
- (NSString *)addBoard:(NSString *)url
                 model:(TEduBoardImageFitMode)model
                  type:(TEduBoardBackgroundType)type
            needSwitch:(BOOL)needSwitch;


/**
 * @brief 增加一页白板
 * @param url                【可选】要使用的背景图片url，为 null 表示不指定背景图片，只支持https协议的图片url
 * @param model              【可选】要使用的图片填充对齐模式`TEduBoardImageFitMode`，当设置url时有效
 * @param type               【可选】背景类型，`TEduBoardBackgroundType`
 * @param needSwitch         【可选】是否切换到添加的白板页
 * @param index              【可选】插入的页面索引
 * @warning 1、白板页会被添加到默认文件（文件 ID 为#DEFAULT)，自行上传的文件无法添加白板页;
 * @warning 2、触发 TEduBoard.EVEN.TEB_ADDBOARD 事件
 *
 * 返回值内存由SDK内部管理，用户不需要自己释放
 */
- (NSString *)addBoard:(NSString *)url
                 model:(TEduBoardImageFitMode)model
                  type:(TEduBoardBackgroundType)type
            needSwitch:(BOOL)needSwitch
                 index:(NSUInteger)index;

/**
 * @brief 删除一页白板
 * @param boardId            要删除的白板 ID，为 nullptr 表示删除当前页
 * @warning 只允许删除默认文件（文件 ID 为#DEFAULT）内的白板页，且默认白板页（白板 ID 为#DEFAULT）无法删除
*/
- (void)deleteBoard:(NSString *)boardId;

/**
 * @brief 上一步
 *  每个 Step 对应 PPT 的一个动画效果，若当前没有已展示的动画效果，则该接口调用会导致向前翻页
*/
- (void)prevStep;

/**
 * @brief 下一步
 *
 * 每个 Step 对应 PPT 的一个动画效果，若当前没有未展示的动画效果，则该接口调用会导致向后翻页
*/
- (void)nextStep;

/**
 * @brief 向前翻页
 *
 * 若当前白板页为当前文件的第一页，则该接口调用无效
*/
- (void)preBoard;

/**
 * @brief 向后翻页
 *
 * 若当前白板页为当前文件的最后一页，则该接口调用无效
*/
- (void)nextBoard;

/**
 * @brief 跳转到指定白板页
 * @param boardId            要跳转到的白板页 ID
 *
 * 允许跳转到任意文件的白板页
*/
- (void)gotoBoard:(NSString *)boardId;

/**
 * @brief 向前翻页
 * @param resetStep            指定翻到指定页以后是否重置 PPT 动画步数
 *
 * 若当前白板页为当前文件的第一页，则该接口调用无效
*/
- (void)preBoard:(BOOL)resetStep;

/**
 * @brief 向后翻页
 * @param resetStep            指定翻到指定页以后是否重置 PPT 动画步数
 *
 * 若当前白板页为当前文件的最后一页，则该接口调用无效
*/
- (void)nextBoard:(BOOL)resetStep;

/**
 * @brief 跳转到指定白板页
 * @param boardId            要跳转到的白板页 ID
 * @param resetStep        指定翻到指定页以后是否重置 PPT 动画步数
 *
 * 允许跳转到任意文件的白板页
*/
- (void)gotoBoard:(NSString *)boardId resetStep:(BOOL)resetStep;

/**
 * @brief 跳转到指定动画步数
 * @param step           动画步数
 * @param boardId        白板id
 *
*/
- (void)gotoStep:(NSInteger)step ofBoard:(NSString *)boardId;

/**
 * @brief 获取当前白板页 ID
 * @return 当前白板页 ID
 *
 * 返回值内存由 SDK 内部管理，用户不需要自己释放
*/
- (NSString *)getCurrentBoard;

/**
 * @brief 获取所有文件的白板列表
 * @return 所有文件的白板列表
*/
- (NSArray<NSString *> *)getBoardList;

/**
 * @brief 设置当前白板页宽高比
 * @param ratio                要设置的白板宽高比
 *
 * 格式如: "4:3"、"16:9"
*/
- (void)setBoardRatio:(NSString *)ratio;

/**
 * @brief 获取当前白板页宽高比
 * @return 白板宽高比，格式与 SetBoardRatio 接口参数格式一致
*/
- (NSString *)getBoardRatio;

/**
 * @brief 设置当前白板页缩放比例
 * @param scale                要设置的白板缩放比例
 *
 * 支持范围: [100，1600]，实际缩放比为: scale / 100
*/
- (void)setBoardScale:(UInt32)scale;

/**
 * @brief 获取当前白板页缩放比例
 * @return 白板缩放比例，格式与 SetBoardScale 接口参数格式一致
*/
- (UInt32)getBoardScale;

/**
 * @brief 获取白板滚动位置 2.6.8新增
 * @return CGPoint的x代表left，y代表top，以百分比形式展示，范围在 [0,1]
*/
- (CGPoint)getBoardScroll;

/**
 * @brief 设置白板内容自适应模式
 * @param mode                要设置的白板内容自适应模式 `TEduBoardContentFitMode`
 *
 * 设置自适应模式后会影响所有后续白板内容操作,受影响接口包括：`[TEduBoardController addTranscodeFile:needSwitch:]`
*/
- (void)setBoardContentFitMode:(TEduBoardContentFitMode)mode;

/**
 * @brief 获取白板内容自适应模式
 * @return 白板内容自适应模式 `TEduBoardContentFitMode`
*/
- (TEduBoardContentFitMode)getBoardContentFitMode;

/**
 * @brief 添加图片资源
 * @param url            【必填】图片地址
 * 支持 png/jpg/gif/svg 格式的本地和网络图片，当 URL 是一个有效的本地文件地址时，该文件会被自动上传到 COS。上传进度回调 onTEBFileUploadProgress，上传结果回调 onTEBFileUploadStatus
*/
- (void)addImageElement:(NSString *)url DEPRECATED_MSG_ATTRIBUTE("从2.6.2起此接口被废弃，请使用addElement添加元素");

/**
 * 添加白板元素， 会触发 `[TEduBoardDelegate onTEBAddElement:url:type:]` 事件
 * @param   url       网页或者图片的 url，只支持 https 协议的网址或者图片 url
 * @param   type     元素类型，当设置TEDU_BOARD_ELEMENT_IMAGE时，等价于addImageElement方法  `TEduBoardElementType`
 * @param   options     配置参数 `TEduBoardAddElementOptions`
 *
 * @return  元素ID
 *
 * @warning （1）当 type = `TEDU_BOARD_ELEMENT_IMAGE`，支持 png、jpg、gif、svg 格式的本地和网络图片，当 url 是一个有效的本地文件地址时，该文件会被自动上传到 COS，上传进度回调 onTEBFileUploadStatus
 * @warning （2）当 type = `TEDU_BOARD_ELEMENT_CUSTOM_GRAPH`，仅支持网络 url，请与自定义图形工具 `TEDU_BOARD_TOOL_TYPE_BOARD_CUSTOM_GRAPH` 配合使用
 * @warning （3）当 type = `TEDU_BOARD_ELEMENT_AUDIO` 或 `TEDU_BOARD_ELEMENT_GLOBAL_AUDIO`，仅支持网络 url
 * @warning （4）addElement不支持涂鸦元素`TEDU_BOARD_ELEMENT_GRAFFITI_LINE` ~ `TEDU_BOARD_ELEMENT_GRAFFITI_GRAPH_OVAL`
 * @warning （5）当 type = `TEDU_BOARD_ELEMENT_MATH_BOARD`, 请使用addElementWithBoard
 * @warning （6）当 type = `TEDU_BOARD_ELEMENT_MATH_GRAPH`, 请使用addElementWithGraph
 * @warning （7）当 type = `TEDU_BOARD_ELEMENT_WATERMARK`, 请使用addElementWithWatermark
 * @warning （8）当 type = `TEDU_BOARD_ELEMENT_FORMULA`, 请使用addElementWithExpression
 */
- (NSString *)addElement:(NSString *)url type:(TEduBoardElementType)type options:(TEduBoardAddElementOptions*)options;
- (NSString *)addElement:(NSString *)url type:(TEduBoardElementType)type;

/**
 * 添加 音频元素，可以添加 title 会触发 `[TEduBoardDelegate onTEBAddElement:url:type:]` 事件
 */
- (NSString *)addAudioElement:(NSString *)audioUrl withTitle:(NSString *)title;
/**
 * 添加公式元素
 * @param formula            【必填】公式 `TEduBoardElementFormula`
 * @param options               配置参数 `TEduBoardAddElementOptions`
 * @return  元素ID
 * @warning 会触发`[TEduBoardDelegate onTEBAddElement:url:type:]`回调
 *
 */
- (NSString *)addElementWithFormula:(TEduBoardElementFormula *)formula options:(TEduBoardAddElementOptions*)options;
/**
 * @brief 添加一个数学函数画板
 * @param mathBoard            【必填】数学函数画板 `TEduBoardElementMathBoard`
 * @param options               配置参数 `TEduBoardAddElementOptions`
 * @return  元素ID
 * @warning 会触发`[TEduBoardDelegate onTEBAddElement:url:type:]`回调
*/
- (NSString *)addElementWithBoard:(TEduBoardElementMathBoard *)mathBoard options:(TEduBoardAddElementOptions*)options;
/**
 * @brief 添加一个数学函数图像
 * @param mathGraph            【必填】数学函数图像 `TEduBoardElementMathGraph`
 * @param options               配置参数 `TEduBoardAddElementOptions`
 * @return  元素ID
 * @warning 会触发 `[TEduBoardDelegate onTEBAddElement:url:type:]`回调
*/
- (NSString *)addElementWithGraph:(TEduBoardElementMathGraph *)mathGraph options:(TEduBoardAddElementOptions*)options;
/**
 * @brief 添加水印
 * @param watermark            【必填】水印参数 `TEduBoardElementWatermark`
 * @param options               配置参数 `TEduBoardAddElementOptions`
 * @return  元素ID
 * @warning 会触发 `[TEduBoardDelegate onTEBAddElement:url:type:]`回调
*/
- (NSString *)addElementWithWatermark:(TEduBoardElementWatermark *)watermark options:(TEduBoardAddElementOptions*)options;

/**
 * @brief 添加一个文本元素
 * @param textContent           文本内容
 * @param fontColor             文本颜色, 默认值"#ff0000"
 * @param fontSize              文本大小, 默认值200
 * @param fontFace              文本字体库, 默认值PingFangSC-Regular
 * @param fontStyle             文本字体样式, `TEduBoardTextStyle`
 * @return  元素ID
*/
- (NSString *)addTextElement:(NSString *)textContent
             fontColor:(NSString *)fontColor
              fontSize:(NSInteger)fontSize
              fontFace:(NSString *)fontFace
             fontStyle:(TEduBoardTextStyle)fontStyle;

/**
 * 删除白板元素
 * @param elementId     元素 ID
 * @return 删除是否成功
 */
- (BOOL)removeElement:(NSString *)elementId;

/**
 * 自动选中元素
 * @param elementId     元素 ID
 */
- (void)autoSelectedElement:(NSString *)elementId;

/**
 * 预设文本工具内容
 * @param input     预设文本内容，取消预设则设置为空
 * @param focus     是否继续保持焦点
 */
- (void)setNextTextInput:(NSString *)input focus:(BOOL)focus;

/**
 * 预设文本工具内容
 * @param zoomIn     放大工具图标 `TEduBoardCursorIcon`
 * @param zoomOut   缩小工具图标 `TEduBoardCursorIcon`
 * @warning 该接口只在桌面端支持
 */
- (void)setZoomCursorIcon:(TEduBoardCursorIcon *)zoomIn zoomOutCursorIcon:(TEduBoardCursorIcon *)zoomOut;

/**
 * @brief 设置白板是否开启笔锋
 * @param enable             【必填】是否开启，true 表示开启，false 表示关闭
 *
 * 白板创建后默认为关闭
*/
- (void)setHandwritingEnable:(BOOL)enable;

/**
 * @brief 获取白板是否开启笔锋
 * @return 是否开启笔锋
*/
- (BOOL)isHandwritingEnable;


/**
 * @brief 刷新当前页白板，触发 onTEBRefresh 回调
 * @warning 如果当前白板包含 PPT/H5/图片/视频时，刷新白板将会触发对应的回调
*/
- (void)refresh;

/**
 * @brief 同步本地发送失败的数据到远端并刷新本地数据
 *
 * 接口用途：此接口主要用于网络恢复后，同步本地数据到远端，拉取远端数据到本地
 * 调用时机：在网络恢复后调用
 * 使用限制：
 * （1）仅支持2.4.9及以上版本
 * （2）如果历史数据还没有加载完成，则不允许重复调用，否则回调告警 TEDU_BOARD_WARNING_ILLEGAL_OPERATION
 * @warning Reload等同于重新加载历史数据，会触发白板初始化时除onTEBInit之外的所有回调。
*/
- (void)syncAndReload;

 /**
  * @brief 白板快照
  * @param info 快照信息  `TEduBoardSnapshotInfo`
 */
- (void)snapshot:(TEduBoardSnapshotInfo *)info;

/**
 * @brief 设置缩放锚点
 * @param xRatio 距离白板左边位置，取值【0, 1】
 * @param yRatio 距离白板顶部位置，取值【0, 1】
*/
- (void)setScaleAnchor:(CGFloat)xRatio yRatio:(CGFloat)yRatio;

/**
 * @brief 设置远端画笔在本地是否可见
 * @param visible  是否可见
 * @param mode  远端画笔操作类型 `TEduBoardRemoteCursorAction`
*/
- (void)setRemoteCursorVisible:(BOOL)visible model:(TEduBoardRemoteCursorAction)mode;

/**
 * @brief 设置缩放工具的缩放比例
 * @param scale  最小缩放步长
*/
- (void)setScaleToolRatio:(NSInteger)scale;

/**
 * @brief 设置橡皮擦单次擦除图层数量
 * @param limit  擦除图层数量，默认为0，即不限制图层数量
*/
- (void)setEraseLayerLimit:(NSInteger)limit;

/**
 * @brief 限制橡皮擦可擦除的白板元素类型
 * @param typeArr  支持设置多个可擦除类型，整型数组 `TEduBoardErasableElementType`
*/
- (void)setEraseLayerType:(NSArray<NSNumber *> *)typeArr;

/**
 * @brief 设置画笔自动拟合模式
 * @param mode     画笔自动拟合模式 `TEduBoardPenFittingMode`
 */
- (void)setPenAutoFittingMode:(TEduBoardPenFittingMode)mode;

/**
 * @brief 开启绘制涂鸦过程中自动拟合图形
 * 用户在绘制过程中，保持鼠标或触摸不抬起会自动触发拟合图形
 * 开启后默认会使用TEduBoardPenFittingMode.AUTO模式拟合
 * @param enable     开启自动拟合
 * @param delay      触发时延，默认2000ms
 * // 示例一：启用自动拟合
 * teduBoard.enablePenAutoFit(true, 2000);
 * // 示例二：修改自动拟合模式
 * teduBoard.enablePenAutoFit(true, 2000);
 * teduBoard.setPenAutoFittingMode(TEduBoardPenFittingMode.RECTANGLE);
 */
- (void)enablePenAutoFit:(BOOL)enable delay:(NSTimeInterval)delay;

/**
 * @brief 设置白板备注
 * @param boardId     【必填】要备注的白板id
 * @param remark      【必填】要备注的内容
 */
- (void)setBoard:(NSString *)boardId remark:(NSString *)remark;

/**
 * @brief 获取白板备注
 * @param boardId     【选填】获取指定的白板备注，不填表示获取所有设置过白板备注
 * @return 白板备注map， key为白板id，value为备注内容
 */
- (NSDictionary<NSString *, NSString *> *)getBoardRemark:(NSString *)boardId;

/**
 * @brief 获取PPT的备注
 * 调用时机：收到PPT加载完成的事件onTEBH5PPTStatusChanged回调状态为`TEDU_BOARD_H5_PPT_STATUS_LOADED`
 * @param fid     【必填】文件ID
 * @param pageIndex 【选填】-1表示获取当前ppt所有备注；其他表示获取当前ppt索引页(从0开始)的备注
 * @return
 * 返回当前页备注 {0: '这是第一页备注'}
 * 返回所有页备注,页数索引从0开始:{0: '这是第一页备注', 1: '这是第二页备注'}
 */
- (NSDictionary<NSString *, NSString *> *)getPPTRemarks:(NSString *)fid pageIndex:(NSInteger)pageIndex;
/**
 * @brief 开启分组模式
 * @param enable     【必填】是否开启
 */
- (void)setClassGroupEnable:(BOOL)enable;

/**
 * @brief 获取分组模式状态
 * @return 是否启用
*/
- (BOOL)getClassGroupEnable;

/**
 * @brief 添加分组
 * @param groupId     【必填】分组id
 * @param boards     【必填】分组白板列表
 * @param users     【必填】分组用户列表
 * @param title     【必填】分组标题
 * @param currentBoardId     【必填】分组初始白板
*/
- (void)setClassGroup:(NSString*)groupId
               boards:(NSArray<NSString *> *)boards
                users:(NSArray<NSString *> *)users
                title:(NSString *)title
       currentBoardId:(NSString *)currentBoardId;

/**
 * @brief 设置分组标题
 * @param groupId     【必填】分组id
 * @param title     【必填】分组标题
*/
- (void)setClassGroup:(NSString*)groupId title:(NSString *)title;

/**
 * @brief 添加白板到分组
 * @param boardId     【必填】白板id
 * @param groupId     【必填】分组id
 */
- (void)addBoard:(NSString *)boardId toClassGroup:(NSString *)groupId;

/**
 * @brief 添加用户到分组
 * @param userId     【必填】白板id
 * @param groupId     【必填】分组id
 */
- (void)addUser:(NSString *)userId toClassGroup:(NSString *)groupId;

/**
 * @brief 获取所有分组id
 * @return     分组id列表
 */
- (NSArray<NSString *> *)getAllClassGroupIds;
/**
 * @brief 重置所有分组
*/
- (void)resetClassGroup;

/**
 * @brief 删除分组
 * @param groupId     【必填】分组id
*/
- (void)removeClassGroup:(NSString *)groupId;

/**
 * @brief 获取分组信息
 * @param groupId     【必填】分组id
 * @return 分组信息 { classGroupId: 分组id, boards: 白板列表, users: 用户列表, currentBoardId: 当前白板id, title: 分组标题 }
*/
- (NSDictionary*)getClassGroupInfoById:(NSString *)groupId;

/**
 * @brief 获取用户所在的分组
 * @param userId     【必填】用户Id
 * @return 分组id列表
*/
- (NSArray<NSString *> *)getClassGroupIdByUserId:(NSString *)userId;

/**
 * @brief 从分组中删除用户
 * @param userId     【必填】用户Id
 * @param groupId     【必填】分组id
*/
- (void)removeUser:(NSString *)userId inClassGroup:(NSString *)groupId;

/**
 * @brief 从分组中删除白板
 * @param boardId     【必填】白板id
 * @param groupId     【必填】分组id
*/
- (void)removeBoard:(NSString *)boardId inClassGroup:(NSString *)groupId;

/**
 * @brief 分组内跳转
 * @param boardId     【必填】白板id
 * @warning 分组模式下切换白板只能通过此方法进行，必须知道需要跳转的白板id
 */
- (void)gotoClassGroupBoard:(NSString *)boardId;

/**
 * @brief 设置白板滚动条是否可见
 * @param enable     【必填】是否可见
 */
- (void)setScrollBarVisible:(BOOL)enable;
/// @}


/// @name 文件操作接口
/// @{

/**
 * @brief 主动查询文件转码进度
 * @param taskId            通过 onTEBFileTranscodeProgress 回调拿到的转码任务 taskId
 * @warning 该接口仅用于特殊业务场景下主动查询文件转码进度，调用 ApplyFileTranscode 后，SDK 内部将会自动定期触发 onTEBFileTranscodeProgress 回调，正常情况下您不需要主动调用此接口
 * @see TEduBoardCallback::onTEBFileTranscodeProgress
 *
 * 转码进度和结果将会通过 onTEBFileTranscodeProgress 回调返回，详情参见该回调说明文档
*/
- (void)getFileTranscodeProgress:(NSString *)taskId;

/**
 * @brief 添加转码文件
 * @param  result             文件转码结果 `TEduBoardTranscodeFileResult`
 * @param  needSwitch    是否跳转到该文件
 * @return 文件ID
 * @warning 当传入文件的 URL 重复时，返回 URL 对应的 文件 ID
 * @warning 在收到对应的 onTEBAddTranscodeFile 回调前，无法用返回的文件 ID 查询到文件信息
 * @see TEduBoardCallback::onTEBAddTranscodeFile
 *
 * TEduBoardTranscodeFileResult 的字段信息主要来自：
 * 1. 使用客户端 ApplyFileTranscode 转码，直接将转码结果用于调用此接口
 * 2.  （推荐）使用服务端 REST API 转码，只需传入转码回调结果的四个字段（title，resolution，url，pages），其服务端->客户端字段的对应关系为 Title->title、Resolution->resolution、ResultUrl->url、Pages->pages [转码文档](https://cloud.tencent.com/document/product/1137/40260)
 *
 * 调用该接口后，SDK 会在后台进行文件加载，期间用户可正常进行其它操作，加载成功或失败后会触发相应回调
 * 文件加载成功后，将自动切换到该文件
*/
- (NSString *)addTranscodeFile:(TEduBoardTranscodeFileResult *)result needSwitch:(BOOL)needSwitch;

/**
 * @brief 删除文件
 * @param fileId            要删除的文件 ID
 *
 * 文件 ID 为 nullptr 时表示当前文件，默认文件无法删除
*/
- (void)deleteFile:(NSString *)fileId;

/**
 * @brief 切换文件
 * @param fileId            要切换到的文件 ID
 * @warning 该接口仅可用于文件切换，如果传入的 fileId 为当前文件 ID，SDK 会忽略其它参数，不做任何操作
 *
 * 文件 ID 为必填项，为 nullptr 或空字符串将导致文件切换失败
*/
- (void)switchFile:(NSString *)fileId;

/**
 * @brief 切换文件
 * @param fileId            要切换到的文件 ID
 * @param boardId           切换文件并跳转到这个白板页
 * @param stepIndex         跳转到白板页并切换到这个动画
 * @warning 该接口仅可用于文件切换，如果传入的 fileId 为当前文件 ID，SDK 会忽略其它参数，不做任何操作
 *
 * 文件 ID 为必填项，为 nullptr 或空字符串将导致文件切换失败
*/
- (void)switchFile:(NSString *)fileId boardId:(NSString *)boardId stepIndex:(NSInteger)stepIndex;

/**
 * @brief 获取当前文件 ID
 * @return 当前文件 ID
*/
- (NSString *)getCurrentFile;

/**
 * @brief 获取白板中指定文件的文件信息
 * @return 文件信息 `TEduBoardFileInfo`
 * @warning 每次调用该接口的返回值内容都指向同一块内存，若需要保存返回信息，请在拿到返回值后及时拷贝走
*/
- (TEduBoardFileInfo *)getFileInfo:(NSString *)fileId;

/**
 * @brief 获取白板中上传的所有文件的文件信息列表
 * @return 文件信息列表  `TEduBoardFileInfo`
 * @warning 返回值不再使用时不需要自行 delete，但是务必调用其 release 方法以释放内存占用
*/
- (NSArray<TEduBoardFileInfo *> *)getFileInfoList;

/**
 * @brief 获取指定文件的白板 ID 列表
 * @param fileId            文件 ID
 * @return 白板 ID 列表
 * @warning 返回值不再使用时不需要自行 delete，但是务必调用其 release 方法以释放内存占用
*/
- (NSArray<NSString *> *)getFileBoardList:(NSString *)fileId;

/**
 * @brief 获取指定文件的缩略图，不支持默认文件（fileId=#DEFAULT）
 * @param fileId            文件 ID
 * @return 缩略图 URL 列表
 *
 * 用户在调用 rest api 请求转码时，需要带上 "thumbnail_resolution" 参数，开启缩略图功能，否则返回的缩略图 url 无效
*/
- (NSArray *)getThumbnailImages:(NSString *)fileId;

/**
 * @brief 设置h5ppt文件 降级渲染
 * @param fileId h5ppt文件 ID
 * @param slideIndex 当前页面索引值
 * @param enable true:降级渲染,false:取消降级
 *
*/
- (void)setDownGrade:(NSString *)fileId slideIndex:(NSInteger)slideIndex enable:(BOOL)enable;
/**
 * @brief 获取白板中所有元素
 * @param boardId   白板 ID，如果为空则获取当前白板所有元素
 * @param includeHideElement 是否包含隐藏元素（使用setElementsDisplay接口隐藏的元素），删除元素不返回（橡皮擦，removeElement接口）
 * @return 白板元素列表   `TEduBoardElementInfo`
*/
- (NSArray<TEduBoardElementInfo *> *)getBoardElementList:(NSString *)boardId includeHideElement:(BOOL)includeHideElement;

/**
 * @brief 设置元素样式
 * @param elementId   元素id
 * @param type  [必填]元素操作类型 元素操作类型 `TEduElementOperatorType`
 * @param value 与元素操作类型对应
 *      TEduElementOperatorTypeBOARDER_COLOR:颜色值,如@"rgba(255, 0, 0, 1)";
 *      TEduElementOperatorTypeBOARDER_WIDTH:  线宽度值, 如@"500";
 *      TEduElementOperatorTypeBOARDER_LINE_TYPE: `TEduBoardLineType` 如TEDU_BOARD_LINE_TYPE_SOLID则为@"1";
 *      TEduElementOperatorTypeEL_DELETE: 传空字符串
 *      TEduElementOperatorTypeCHANGE_Text_Size: 文本元素字体大小, 如@"800";
 *      TEduElementOperatorTypeRotationAngle: 传整数 0~360
 *
*/
- (void)updateElementById:(NSString *)elementId type:(TEduElementOperatorType)type value:(NSString *)value;

/**
 * @brief 设置元素的位置，TEduElementOperatorTypeCHANGE_POSITION
 * @param elementId   元素id
 * @param offset 偏移量 元素在x、y轴上的偏移
 * offset.x为负数表示左移，正数表示右移
 * offset.y为负数表示上移，正数表示下移
 *
*/
- (void)updateElementById:(NSString *)elementId position:(CGPoint)offset;

/**
   * @brief 复制元素，仅支持几何图形
   * @param elementIds 元素id集合
   */
- (void)copyElementsByIds:(NSArray<NSString *> *)elementIds;

/**
   * @brief 复制元素，仅支持几何图形
   * @param elementIds 元素id集合
   * @param styleObjects 自定义元素样式集合
   * @param config 配置参数对象
   */
- (void)copyElementsByIds:(NSArray<NSString *> *)elementIds
             styleObjects:(NSArray<TEduBoardCopyGraphElementStyle *> *)styleObjects
                   config:(TEduBoardCopyGraphElementConfig *)config;

/**
 * @brief 获取白板中的元素信息
 * @param elementId   元素id
 * @return 白板元素信息   `TEduBoardElement`
*/
- (TEduBoardElement *)getElementById:(NSString *)elementId;
/**
 * @brief 清空指定文件的所有白板涂鸦
 * @param fileId            文件 ID
*/
- (void)clearFileDraws:(NSString *)fileId;

/**
 * @brief 添加视频文件
 * @param url         文件播放地址
 * @param title                 文件名
 * @param needSwitch            添加文件后是否切换到该文件
 * @return 文件 ID
 * @warning 当传入文件的 URL 重复时，返回 URL 对应的 文件 ID
 *
 * 支持 mp4/m3u8/hls，触发状态改变回调 onTEBVideoStatusChanged
 */
- (NSString *)addVideoFile:(NSString *)url title:(NSString *)title needSwitch:(BOOL)needSwitch;

/**
 * @brief 显示或隐藏视频控制栏
 * @param show        是否显示
 * @warning 全局控制项，对所有视频文件有效
 * 隐藏和显示默认视频控制栏，默认显示系统自带的 video 控制栏，不同平台界面 UI 样式不同
 */
- (void)showVideoControl:(BOOL)show;

/**
 * @brief 播放视频
 * @warning 只对当前文件有效
 *
 * 触发状态改变回调 onTEBVideoStatusChange，一般在使用自定义视频控制栏时使用
 * 移动端回前台调用 play（WebView 默认行为）
 */
- (void)playVideo;

/**
 * @brief 暂停视频
 * @warning 只对当前文件有效
 *
 * 触发状态改变回调 onTEBVideoStatusChanged，一般在使用自定义视频控制栏时使用
 * 移动端退后台调用 pause（WebView 默认行为）
*/
- (void)pauseVideo;

/**
 * @brief 跳转（仅支持点播视频）
 * @param time 播放进度，单位秒
 * @warning 只对当前文件有效
 *
 * 触发状态改变回调 onTEBVideoStatusChanged，一般在使用自定义视频控制栏时使用
 */
- (void)seekVideo:(CGFloat)time;

/**
 * @brief 设置PPT静音播放
 * @param mute true 静音 false 取消静音
 * @warning 1、只能对本地生效，不会影响远端
 * @warning 2、只对旧转码方案生效
 * @warning 3、调用时机请在ppt课件加载状态事件中收到加载完成后再调用：
 * -(void)onTEBH5PPTStatusChanged:(NSString *)fileId status:(TEduBoardH5PPTStatus)status message:(NSString *)message;中收到TEDU_BOARD_H5_PPT_STATUS_LOADED
 */
- (void)soundMuteForPPT:(BOOL)mute;
/**
 * @brief 是否同步本地视频操作到远端
 * @param enable            【必填】是否同步
 * @warning 全局控制项，对所有视频文件有效
 *
 * play/pause/seek 接口以及控制栏事件的触发是否影响远端，默认为 true
 * 一般情况下学生设置为 false，老师设置为 true
 */
- (void)setSyncVideoStatusEnable:(BOOL)enable;

/**
 * @brief 是否同步本地视频操作到远端
 * @param enable            【必填】是否同步
 * @param isAppActiveChange 【选填】只有移动端在app切换前后台的时候需要设置，其他端不需要设置
 * @warning 全局控制项，对所有视频文件有效
 *
 * play/pause/seek 接口以及控制栏事件的触发是否影响远端，默认为 true
 * 一般情况下学生设置为 false，老师设置为 true
 */
- (void)setSyncVideoStatusEnable:(BOOL)enable isAppActiveChange:(BOOL)isAppActiveChange;

/**
 * @brief 内部启动定时器，定时同步视频状态到远端（仅限于 mp4）
 * @param interval            【选填】同步间隔，例如设置5秒
 * @warning 只对当前文件有效
 *
 * 一般在老师端视频加载完成后调用，切换文件后内部自动销毁定时器，
 */
- (void)startSyncVideoStatus:(NSInteger)interval;

/**
 * @brief 停止同步视频状态
 * @warning 只对当前文件有效
 */
- (void)stopSyncVideoStatus;

/**
 * @brief 是否同步本地音频操作到远端
 * @param enable            【必填】是否同步
 * @warning 全局控制项，对所有视频文件有效
 *
 * play/pause/seek 接口以及控制栏事件的触发是否影响远端，默认为 true
 * 一般情况下学生设置为 false，老师设置为 true
 */
- (void)setSyncAudioStatusEnable:(BOOL)enable;

/**
 * @brief 是否启用音频控制面板
 * @param enable        启用或禁止
 * @warning 禁止控制面板后，不能通过界面交互方式操作音频元素
 */
- (void)enableAudioControl:(BOOL)enable;

/**
 * @brief 播放音频
 * @param elementId     调用 addElement 方法返回的元素 ID
 *
 * 触发状态改变回调 `[TEduBoardDelegate onTEBAudioStatusChanged:status:progress:duration:]`
 */
- (void)playAudio:(NSString *)elementId;

/**
 * @brief 暂停音频
 * @param elementId     调用 addElement 方法返回的元素 ID
 *
 * 触发状态改变回调 `[TEduBoardDelegate onTEBAudioStatusChanged:status:progress:duration:]`
 */
- (void)pauseAudio:(NSString *)elementId;

/**
 * @brief 跳转
 * @param elementId     调用 addElement 方法返回的元素 ID
 * @param time                播放进度，单位秒
 *
 * 触发状态改变回调 `[TEduBoardDelegate onTEBAudioStatusChanged:status:progress:duration:]`
 */
- (void)seekAudio:(NSString *)elementId time:(CGFloat)time;

/**
 * @brief 设置音量
 * @param elementId     调用 addElement 方法返回的元素 ID
 * @param volume  音频音量，取值范围[0-1]
 */
- (void)setAudioVolume:(NSString *)elementId volume:(CGFloat)volume;
 
 /**
  * 获取音量
  * @param elementId     调用 addElement 方法返回的元素 ID
  * @return 当前音量，取值范围[0-1]
  */
- (CGFloat)getAudioVolume:(NSString *)elementId;

/**
 * 播放ppt视频
 * @param videoIDs 视频元素id数组
 *
 */
- (void)playH5PPTVideo:(NSArray<NSString *> *)videoIDs;

/**
 * 播放ppt音频
 * @param audioIDs 音频元素id数组
 *
 */
- (void)playH5PPTAudio:(NSArray<NSString *> *)audioIDs;

/**
 * 暂停ppt视频
 * @param videoIDs 视频元素id数组
 *
 */
- (void)pauseH5PPTVideo:(NSArray<NSString *> *)videoIDs;

/**
 * 暂停ppt音频
 * @param audioIDs 音频元素id数组
 */
- (void)pauseH5PPTAudio:(NSArray<NSString *> *)audioIDs;

/**
 * @brief 添加 H5 页面
 * @param url                   网页地址
 * @param title                 文件名
 * @param needSwitch            添加文件后是否切换到该文件
 * @return 文件 ID
 * @warning 只支持展示，不支持互动
 */
- (NSString *)addH5File:(NSString *)url title:(NSString *)title needSwitch:(BOOL)needSwitch;

/**
 * @brief 批量导入图片到白板
 * @param urls                  要使用的背景图片 URL 列表，编码格式为 UTF8
 * @param title                 文件名
 * @param needSwitch            添加文件后是否切换到该文件
 * @return 新增加文件Id
 * @warning 当传入文件的 URL 重复时，返回 URL 对应的 文件 ID
 */
- (NSString *)addImagesFile:(NSArray<NSString *> *)urls title:(NSString *)title needSwitch:(BOOL)needSwitch;

/**
 * @brief 调用此接口可在后台生成当前白板的板书内容
 */
- (void)addSnapshotMark;

/**
 * @brief 用于课件预加载，减少课件的加载时间
 * @warning iOS 11.4、macOS 11.0以上系统有效
 */
- (BOOL)loadResource:(NSString *)zipUrl;

/**
 * @brief 导出数据
 */
- (NSString *)exportInLocalMode;

/**
 * @brief 导入数据
 * @param data  通过exportInLocalMode导出的数据
 */
- (void)importInLocalMode:(NSString *)data;

/**
 * @brief 导出数据
 * 1. exclude是指导出的数据中不包含该元素
 * 2. hide是指导出的数据中包含该元素, 但将其设置为隐藏状态
 * 3. 如果filter中对应的值都没有设置或者设置空数组，则不进行过滤
 * @param filter    Json字符串，具体格式如下
 * filter.include
 * filter.include.types 要包含的元素类型 {@link TEduBoardElementType}
 * filter.include.ids  要包含的元素
 * filter.exclude
 * filter.exclude.types  要排除的元素类型 {@link TEduBoardElementType}
 * filter.exclude.ids  要排除的元素
 * filter.hide
 * filter.hide.types 要隐藏的元素类型 {@link TEduBoardElementType}
 * filter.hide.ids 要隐藏的元素
 */
- (NSString *)exportInLocalModeWithFilter:(NSString *)filter;

/**
 * @brief 导入数据
 * @param data  通过exportInLocalMode导出的数据
 * @param filter  过滤器，具体请参考方法 {@link exportInLocalModeWithFilter}
 */
- (void)importInLocalMode:(NSString *)data withFilter:(NSString *)filter;

/**
 * @brief 设置元素显示状态
 * 1. 如果元素类型和元素id都不设置，或者设置为空数组，则表示全部元素
 * 2. 如果元素类型和元素id都设置，则取其符合条件元素的并集
 * @param types 元素类型数组 {@link TEduBoardElementType}
 * @param ids 元素ID数组
 * @param visible 是否显示
 */
- (void)setElementsDisplay:(NSArray<NSNumber *>*)types ids:(NSArray<NSString*>*)ids visible:(BOOL)visible;

/**
 * @brief 锁定元素
 * @param elementIds ID 列表
 * @param locked          是否锁定
 */
- (void)lockElements:(NSArray<NSString *>*)elementIds locked:(BOOL)locked;
/// @}
@end

